package ru.yandex.calendar.logic.log.dao;

import java.sql.SQLException;
import java.time.Duration;
import java.time.Instant;

import com.fasterxml.jackson.core.JsonProcessingException;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.postgresql.util.PGobject;

import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.log.LogEvent;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.commune.dynproperties.DynamicProperty;

@Slf4j
public class EventsLogDao extends CalendarJdbcDaoSupport {
    private final DynamicProperty<Long> defaultTtlMinutes = new DynamicProperty<>("eventsLogDefaultTtlMinutes", Duration.ofDays(28).toMinutes());

    public void logSafe(LogEvent event, ActionInfo actionInfo) {
        logSafe(event, actionInfo, Duration.ofMinutes(defaultTtlMinutes.get()));
    }

    public void logSafe(LogEvent event, ActionInfo actionInfo, Duration ttl) {
        try {
            logUnsafe(event, actionInfo, ttl);
        } catch (SQLException | JsonProcessingException | RuntimeException e) {
            log.error(String.format("Can't log event by type %s with action %s", event.type(), actionInfo.getAction()), e);
        }
    }

    public void logUnsafe(LogEvent event, ActionInfo actionInfo, Duration ttl) throws SQLException, JsonProcessingException {
        val q = "INSERT INTO events_log " +
                "(action, action_source, req_id, host_id, tvm_id, event_id, main_event_id, recurrence_id, data, action_ts, record_expiration_ts) " +
                "VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

        val data = new PGobject();
        data.setType("json");
        data.setValue(EventsLogger.mapper.writeValueAsString(event));

        getJdbcTemplate().update(
                q,
                actionInfo.getAction(),
                actionInfo.getActionSource(),
                actionInfo.getRequestId(),
                actionInfo.getRequestIdWithHostId(),
                actionInfo.getTvmId().stream().boxed().findAny().orElse(null),
                event.getEventId().map(e -> e.eventId).orElse(null),
                event.getEventId().map(e -> e.mainEventId).orElse(null),
                event.getEventId().flatMap(e -> e.recurrenceId).orElse(null),
                data,
                actionInfo.getNow(),
                Instant.now().plus(ttl)
        );
    }

    public int gc() {
        return getJdbcTemplate().update("DELETE FROM events_log WHERE record_expiration_ts < now()");
    }
}
