package ru.yandex.calendar.logic.log.requests;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNullableByDefault;
import javax.servlet.http.HttpServletRequest;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import org.joda.time.Instant;
import org.springframework.web.util.ContentCachingRequestWrapper;
import org.springframework.web.util.WebUtils;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.ByteArrayInputStreamSource;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.net.HostnameUtils;

public class RequestsLogger {

    public static final String NAME = "requests-json";

    private static final String HOST = StringUtils.substringBefore(
            HostnameUtils.localHostname(), ".qloud-c.yandex.net");

    private static final Logger logger = LoggerFactory.getLogger(NAME);

    private static final ObjectMapper mapper = new ObjectMapper()
            .registerModule(new JodaModule())
            .configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false)
            .setVisibility(PropertyAccessor.FIELD, JsonAutoDetect.Visibility.ANY);

    public static void log(HttpServletRequest request, PassportUid uid) {
        ContentCachingRequestWrapper caching = WebUtils.getNativeRequest(
                request, ContentCachingRequestWrapper.class);

        if (caching != null) {
            log(caching, uid, Option.empty());
        }
    }

    public static void log(
            ContentCachingRequestWrapper request,
            PassportUid uid, Option<IcsCalendar> ics
    ) {
        String query = request.getQueryString();
        String uri = request.getRequestURI() + (query == null ? "" : "?" + query);

        String encoding = ObjectUtils.defaultIfNull(request.getRequest().getCharacterEncoding(), "UTF-8");
        String body = new ByteArrayInputStreamSource(request.getContentAsByteArray()).readText(encoding);

        LogEntry entry = new LogEntry(
                Instant.now(), uid.getUid(),
                HOST, request.getLocalPort(),
                RequestIdStack.current().orElse((String) null),
                request.getMethod(), uri,  request.getContentType(), body.isEmpty() ? null : body,
                ics.map(IcsCalendar::serializeToString).orElse((String) null));
        try {
            logger.info(mapper.writeValueAsString(entry));
        } catch (JsonProcessingException e) {
            throw IoUtils.translate(e);
        }
    }

    @ParametersAreNullableByDefault
    @SuppressWarnings("ALL")
    private static class LogEntry {
        private final Instant dt;
        private final long ts;
        private final long uid;
        private final String host;
        private final int port;
        private final String rid;
        private final String method;
        private final String uri;
        private final String contentType;
        private final String body;
        private final String ics;

        public LogEntry(
                @Nonnull Instant dt, long uid,
                String host, int port, String rid,
                String method, String uri,
                String contentType, String body, String ics
        ) {
            this.dt = dt;
            this.ts = dt.getMillis();
            this.uid = uid;
            this.host = host;
            this.port = port;
            this.rid = rid;
            this.method = method;
            this.uri = uri;
            this.contentType = contentType;
            this.body = body;
            this.ics = ics;
        }
    }
}
