package ru.yandex.calendar.logic.mailer;

import lombok.val;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.generated.MailerEvent;
import ru.yandex.calendar.logic.contact.UnivContact;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActorId;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventTime;
import ru.yandex.calendar.logic.event.dao.EventResourceDao;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.ics.exp.EventInstanceParameters;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsVTimeZones;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsMethod;
import ru.yandex.calendar.logic.log.EventIdLogDataJson;
import ru.yandex.calendar.logic.mailer.change.MailerEventChange;
import ru.yandex.calendar.logic.mailer.model.FoundEvent;
import ru.yandex.calendar.logic.mailer.model.MailerMail;
import ru.yandex.calendar.logic.mailer.model.MailerParticipant;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.sending.param.CancelEventMessageParameters;
import ru.yandex.calendar.logic.sending.param.CommonEventMessageParameters;
import ru.yandex.calendar.logic.sending.param.EventInviteeNamesI18n;
import ru.yandex.calendar.logic.sending.param.EventLocation;
import ru.yandex.calendar.logic.sending.param.EventMessageChanged;
import ru.yandex.calendar.logic.sending.param.EventMessageInfo;
import ru.yandex.calendar.logic.sending.param.EventMessageParameters;
import ru.yandex.calendar.logic.sending.param.EventTimeParameters;
import ru.yandex.calendar.logic.sending.param.InvitationMessageParameters;
import ru.yandex.calendar.logic.sending.param.Recipient;
import ru.yandex.calendar.logic.sending.param.ReplyMessageParameters;
import ru.yandex.calendar.logic.sending.param.Sender;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.MailType;
import ru.yandex.calendar.logic.sharing.ReplyInfo;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.svc.SvcRoutines;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.logic.user.NameI18nWithOptionality;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.logic.user.UserOrMaillist;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.commune.mail.MailAddress;
import ru.yandex.inside.passport.PassportSid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author dbrylev
 */
public class MailerEventMessageCreator {

    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private UserManager userManager;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private EventResourceDao eventResourceDao;
    @Autowired
    private SvcRoutines svcRoutines;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;


    public EventMessageParameters createRequestOrCancelMessage(
            UserOrMaillist user, MailerEventChange change, MailerMail mail, FoundEvent foundEvent)
    {
        MailerEvent event = change.getCurrentOrPrevious();
        BaseParams params = createBaseParams(user, event, mail, foundEvent);

        EventMessageInfo messageInfo = createEventMessageInfo(params.eventId, event, params);
        CommonEventMessageParameters commonParams = createCommonMessageParameters(user, event, mail, params);

        commonParams = commonParams.withHiddenActor();

        Option<IcsCalendar> ics = Option.of(mail.ics);

        if (change.isCancelled()) {
            return new CancelEventMessageParameters(commonParams, messageInfo, ics, MailType.EVENT_CANCEL);
        }

        EventInviteeNamesI18n guests = new EventInviteeNamesI18n(
                change.getCurrentUserAttendees().map(this::getInviteeName),
                change.getNewUserAttendees().map(this::getInviteeName),
                change.getRemovedUserAttendees().map(this::getInviteeName));

        Option<EventMessageChanged> changed = Option.when(change.isUpdated(), change::findChanged);

        return new InvitationMessageParameters(
                commonParams, messageInfo, ics, Option.empty(), params.privateToken, guests, changed, params.decision,
                false, change.isUpdated() ? MailType.EVENT_UPDATE : MailType.EVENT_INVITATION);
    }

    public EventMessageParameters createReplyMessage(
            UserOrMaillist user, MailerEvent event, ReplyInfo reply, MailerMail mail, FoundEvent foundEvent)
    {
        BaseParams params = createBaseParams(user, event, mail, foundEvent);

        EventMessageInfo messageInfo = createEventMessageInfo(params.eventId, event, params);
        CommonEventMessageParameters commonParams = createCommonMessageParameters(user, event, mail, params);

        return new ReplyMessageParameters(
                commonParams, messageInfo,
                reply.getDecision(), reply.getReason(), Option.of(mail.ics), MailType.EVENT_REPLY);
    }

    public BaseParams createBaseParams(UserOrMaillist user, MailerEvent event, MailerMail mail, FoundEvent foundEvent) {

        Option<SettingsInfo> recipientSettings = settingsRoutines.getSettingsByUidIfExists(event.getUid());
        Option<SettingsInfo> senderSettings = mail.sender.getUid().map(settingsRoutines::getSettingsByUid);

        Language lang = recipientSettings.plus(senderSettings)
                .map(s -> s.getCommon().getLanguage()).firstO().getOrElse(Language.RUSSIAN);

        DateTimeZone userTz = recipientSettings.map(SettingsInfo::getTz)
                .getOrElse(() -> DateTimeZone.forID(event.getTimezoneId()));

        EventTime time = new EventTime(event.getStartTs(), event.getEndTs(), event.getIsAllDay(), event.getTimezoneId());

        EventInstanceParameters instance = new EventInstanceParameters(
                event.getStartTs(), event.getEndTs(), event.getRecurrenceId());

        ListF<Instant> exdates = mail.ics.getEvents().flatMap(e -> e.getExDates()
                .flatMap(es -> es.getInstants(IcsVTimeZones.fallback(time.getTz()))));

        RepetitionInstanceInfo repetition = new RepetitionInstanceInfo(
                instance.getInterval(), time.getTz(),
                event.getRepetition().map(r -> r.toRepetition(time.getStart())),
                Cf.list(), exdates.map(RepetitionUtils::consExdate), Cf.list());

        EventTimeParameters timeParams = EventTimeParameters.create(time, instance, repetition, userTz);

        val foundEventId = foundEvent.id.getOrElse(
                () -> new EventIdLogDataJson(0, 0, event.getExternalId(), event.getRecurrenceId().toOptional()));

        Decision decision = foundEvent.decision.getOrElse(Decision.UNDECIDED);
        Option<String> privateToken = foundEvent.privateToken;

        Recipient recipient = Recipient.of(new MailAddress(user.email), recipientSettings);

        return new BaseParams(recipient, foundEventId, timeParams, mail.ics.getMethod(), decision, privateToken, userTz, lang);
    }

    public CommonEventMessageParameters createCommonMessageParameters(
            UserOrMaillist user, MailerEvent event, MailerMail mail, BaseParams params)
    {
        ParticipantId senderId = eventInvitationManager.getParticipantIdByEmail(mail.sender.getEmail());

        Sender sender = senderId.getSubjectId()
                .map(id -> eventInvitationManager.getSender(ActorId.userOrResource(id)))
                .getOrElse(mail.sender::asSender);

        Email emailFrom = svcRoutines.getCalendarInfoEmail(event.getUid());

        String calendarUrl = svcRoutines.getCalendarUrlForDomain(PassportAuthDomain.byUid(event.getUid()));

        return new CommonEventMessageParameters(
                params.lang, new LocalDateTime(params.userTz), sender, params.recipient, emailFrom,
                calendarUrl, false, mail.toOverrides(user));
    }

    public EventMessageInfo createEventMessageInfo(
            EventIdLogDataJson eventId, MailerEvent event, BaseParams params)
    {
        Language lang = params.lang;

        ListF<Long> resourceIds = params.method.sameMethodAs(IcsMethod.REPLY)
                ? eventResourceDao.findEventResourceIdsByEventIds(Cf.list(params.eventId.eventId))
                : MailerEventChange.getResourceIds(event);

        ListF<ResourceInfo> resources = resourceDao.findResourceInfosByIds(resourceIds);

        if (resources.isNotEmpty()) {
            resources = resourceRoutines.sortResourcesFromUserOfficeAndCityFirst(event.getUid(), resources);
        }

        EventLocation location = !passportAuthDomainsHolder.containsYandexTeamRu()
                ? EventLocation.compoundForPublic(event.getLocation().orElse(""), resources, Cf.list(), lang)
                : resources.isEmpty()
                ? EventLocation.location(event.getLocation().getOrElse(""))
                : event.getUid().isYandexTeamRu()
                ? EventLocation.resourcesWithStaffMapLinkAndPhones(resources, Cf.list(), lang)
                : EventLocation.resources(resources, Cf.list(), lang);

        Option<UnivContact> organizer = event.getOrganizer().map(p ->
                new UnivContact(p.getEmail(), getInviteeName(p).getNameI18n().getName(lang), p.getId().getUidIfYandexUser()));

        return new EventMessageInfo(
                eventId.eventId, eventId.mainEventId, eventId.externalId, Option.ofNullable(eventId.recurrenceId.orElse(null)),
                params.time, event.getName(), event.getDescription().getOrElse(""),
                location, PassportSid.CALENDAR, organizer, Option.empty());
    }

    public NameI18nWithOptionality getInviteeName(MailerParticipant participant) {
        if (participant.getId().getUidIfYandexUser().exists(PassportUid::isYandexTeamRu)) {
            NameI18n name = userManager.getYtUserNameByEmail(participant.getEmail())
                    .orElseGet(() -> new NameI18n(Emails.getUnicoded(participant.getEmail()), Option.empty()));
            return new NameI18nWithOptionality(name, false);
        } else {
            String ru = StringUtils.defaultIfEmpty(
                    participant.getNameIfExternal().getOrNull(), Emails.getUnicoded(participant.getEmail()));

            NameI18n name = new NameI18n(ru, Option.empty());
            return new NameI18nWithOptionality(name, false);
        }
    }

    public static class BaseParams {
        public final Recipient recipient;

        public final EventIdLogDataJson eventId;
        public final EventTimeParameters time;

        public final IcsMethod method;

        public final Decision decision;
        public final Option<String> privateToken;

        public final DateTimeZone userTz;
        public final Language lang;

        public BaseParams(
                Recipient recipient, EventIdLogDataJson eventId,
                EventTimeParameters time, IcsMethod method,
                Decision decision, Option<String> privateToken,
                DateTimeZone userTz, Language lang)
        {
            this.recipient = recipient;
            this.eventId = eventId;
            this.time = time;
            this.method = method;
            this.decision = decision;
            this.privateToken = privateToken;
            this.userTz = userTz;
            this.lang = lang;
        }
    }
}
