package ru.yandex.calendar.logic.mailer;

import java.util.Set;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.api.mail.MailEventException;
import ru.yandex.calendar.frontend.api.mail.MailEventManager;
import ru.yandex.calendar.frontend.mulca.Mulcagate;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.MailerEvent;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.RecurrenceIdOrMainEvent;
import ru.yandex.calendar.logic.event.SequenceAndDtStamp;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsVEventGroup;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsMethod;
import ru.yandex.calendar.logic.log.EventIdLogDataJson;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.mailer.change.MailerEventChange;
import ru.yandex.calendar.logic.mailer.change.MailerGroupChangesInfo;
import ru.yandex.calendar.logic.mailer.logbroker.MailAttach;
import ru.yandex.calendar.logic.mailer.model.FoundEvent;
import ru.yandex.calendar.logic.mailer.model.MailerMail;
import ru.yandex.calendar.logic.mailer.model.MailerParticipant;
import ru.yandex.calendar.logic.sending.real.MailHeaders;
import ru.yandex.calendar.logic.sending.real.MailSender;
import ru.yandex.calendar.logic.sharing.ReplyInfo;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.UserParticipantInfo;
import ru.yandex.calendar.logic.update.LockResource;
import ru.yandex.calendar.logic.update.LockTransactionManager;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.logic.user.UserOrMaillist;
import ru.yandex.calendar.micro.yt.entity.YtUser;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.mail.MailMessage;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.mail.cerberus.yt.data.YtUserInfo;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.lang.StringUtils;


@Slf4j
public class MailerHandler {
    private final DynamicProperty<Integer> waitForMaster = new DynamicProperty<>("mailerWaitForMasterSeconds", 60);

    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private MailerEventMessageCreator messageCreator;
    @Autowired
    private MailerEventImporter mailerEventImporter;
    @Autowired
    private UserManager userManager;
    @Autowired
    private Mulcagate mulcagate;
    @Autowired
    private MailSender sender;
    @Autowired
    private LockTransactionManager lockTransactionManager;
    @Autowired
    private EventsLogger eventsLogger;

    public void handleAttach(MailAttach attach, ActionInfo actionInfo) {
        String text = mulcagate.getMail(attach.stid);
        MailMessage message = MailMessage.parse(InputStreamSourceUtils.bytes(text.getBytes()));

        if (!message.getHeader(MailHeaders.Y_EXCHANGE_CALENDAR).isSome("Yes")
                || message.getHeader(MailHeaders.X_MAILER).isSome(MailHeaders.YA_CALENDAR))
        {
            return;
        }
        UserOrMaillist user = resolveUser(attach.uid);

        MailerHandlingLogEventJson result;

        if (message.getHeader(MailHeaders.IN_REPLY_TO).isPresent()) {
            result = new MailerHandlingLogEventJson(user, attach.mid, message, MailerHandlingResult.returned("forwarded"));
        } else {
            result = MailerMail.parseOrRefusal(message, attach).fold(
                    m -> handleMail(user, m, actionInfo),
                    r -> new MailerHandlingLogEventJson(user, attach.mid, message, MailerHandlingResult.returned(r)));
        }
        log.info("event: " + result.getEventId() + ", status:" + result.status + ", reason: " + result.getReason());
        if (result.status == MailerHandlingResult.Status.RETURNED) {
            sender.send(user.email, MailMessageUtils.transformForReturn(user, message, actionInfo));
        }
        eventsLogger.log(result, actionInfo);
    }

    private MailerHandlingLogEventJson handleMail(UserOrMaillist user, MailerMail mail, ActionInfo actionInfo) {
        try {
            val group = MailEventManager.getAcceptableGroup(mail.ics, Set.of(IcsMethod.PUBLISH, IcsMethod.REQUEST, IcsMethod.CANCEL, IcsMethod.REPLY));
            val method = mail.ics.getMethod();
            val result = method.sameAs(IcsMethod.REPLY)
                    ? handleReply(user, mail, group, actionInfo)
                    : handleRequestOrCancellation(user, mail, group, actionInfo);
            return new MailerHandlingLogEventJson(user, mail, result);
        } catch (MailEventException e) {
            return new MailerHandlingLogEventJson(user, mail, MailerHandlingResult.returned(e.getMessage()));
        }
    }

    private MailerHandlingResult handleRequestOrCancellation(
            UserOrMaillist user, MailerMail mail, IcsVEventGroup group, ActionInfo actionInfo)
    {
        MailerGroupChangesInfo changes = mailerEventImporter.findChanges(user.uid, mail.ics, actionInfo);

        if (changes.isEmpty()) {
            return MailerHandlingResult.skipped("outdated");
        }

        MailerEvent mailerEvent = changes.getForMail().getCurrentOrPrevious();
        FoundEvent calendarEvent = findCalendarEvent(user, group, mailerEvent, mail.ics.getMethod());

        if (!mail.ics.getMethod().sameAs(IcsMethod.CANCEL)) {
            if (calendarEvent.isDeleted()) {
                return MailerHandlingResult.skipped("deleted");
            }
            if (!calendarEvent.isFound()) {
                throw new RuntimeException("Event not found in calendar " + mailerEvent.getId());
            }
            if (!calendarEvent.isInvitationFound()) {
                throw new RuntimeException("Event invitation not found in calendar " + mailerEvent.getId());
            }
            if (calendarEvent.isMasterAttendee()
                    && mailerEvent.getRecurrenceId().isPresent() && changes.getForMail().isCreated()
                    && mail.date.isAfter(actionInfo.getNow().minus(Duration.standardSeconds(waitForMaster.get()))))
            {
                throw new RuntimeException("Waiting master event to be processed for " + mailerEvent.getId());
            }
        }

        return lockTransactionManager.lockAndDoInTransaction(LockResource.mailerUser(user.uid), () -> {
            MailerEventChange change = mailerEventImporter.importIcs(user.uid, mail.ics, calendarEvent, actionInfo);

            if (change.isIgnored()) {
                return MailerHandlingResult.skipped("outdated");

            } else if (isPast(change.getCurrentOrPrevious(), group, actionInfo)) {
                return MailerHandlingResult.skipped("past");

            } else if (user.getUidIfUser()
                    .filterMap(uid -> Option.x(userManager.getYtUserByUid(uid)))
                    .map(YtUser::getInfo)
                    .exists(YtUserInfo::isDismissed)) {
                return MailerHandlingResult.skipped("dismissed");
            } else {
                eventInvitationManager.sendEventMails(Cf.list(messageCreator.createRequestOrCancelMessage(
                        user, change, mail.forSending(), calendarEvent)), actionInfo);

                return MailerHandlingResult.sent();
            }
        });
    }

    private MailerHandlingResult handleReply(
            UserOrMaillist user, MailerMail mail, IcsVEventGroup group, ActionInfo actionInfo)
    {
        MailerEvent event = mailerEventImporter.convert(user.uid, mail.ics, actionInfo).getMasterOrFirstEvent();

        if (event.getAttendees().attendees.size() != 1) {
            return MailerHandlingResult.returned("attendee-mismatch");
        }
        if (isPast(event, group, actionInfo)) {
            return MailerHandlingResult.skipped("past");
        }

        ReplyInfo reply = new ReplyInfo(
                event.getAttendees().attendees.single().decision,
                group.getMasterOrFirstEvent().getComment().map(StringUtils::trimToNull).filterNotNull(),
                new SequenceAndDtStamp(event.getSequence(), event.getDtstamp()));

        FoundEvent calendarEvent = findCalendarEvent(user, group, event, mail.ics.getMethod());

        eventInvitationManager.sendEventMails(Cf.list(
                messageCreator.createReplyMessage(user, event, reply, mail.forSending(), calendarEvent)), actionInfo);

        return MailerHandlingResult.sent();
    }

    private boolean isPast(MailerEvent event, IcsVEventGroup group, ActionInfo actionInfo) {
        return group.getEvents().size() == 1
                && event.getRecurrenceId().exists(actionInfo.getNow()::isAfter)
                && event.getEndTs().isBefore(actionInfo.getNow());
    }

    private UserOrMaillist resolveUser(PassportUid uid) {
        return userManager.getEmailByUid(uid)
            .map(e -> UserOrMaillist.user(uid, e))
            .orElseGet(() -> UserOrMaillist.maillist(uid, userManager.getEmailByUidDirectly(uid)));
    }

    private FoundEvent findCalendarEvent(
            UserOrMaillist user, IcsVEventGroup group, MailerEvent mailerEvent, IcsMethod method)
    {
        Option<MainEvent> mainEvent = eventRoutines.getMainEventBySubjectIdAndParticipantEmailsAndExternalId(
                user.getSubjectId(), group.getParticipantEmailsSafe(), group.getUid().get());

        if (mainEvent.isPresent()) {
            Option<Instant> recurrenceId = mailerEvent.getRecurrenceId();

            Option<Event> event = eventRoutines.findEventByMainEventIdAndRecurrence(
                    mainEvent.get().getId(), new RecurrenceIdOrMainEvent(recurrenceId));

            if (event.isPresent()) {
                Option<UserParticipantInfo> participant = Option.empty();
                Option<UserParticipantInfo> masterParticipant = Option.empty();

                if (method.sameMethodAs(IcsMethod.REQUEST)) {
                    ListF<ParticipantId> possibleIds;
                    ListF<MailerParticipant> attendees = mailerEvent.getAttendees().attendees;

                    if (!attendees.map(MailerParticipant::getId).exists(user.getPossibleIds()::containsTs)) {
                        ListF<ParticipantId> maillists = attendees
                                .filterMap(MailerParticipant::getEmailIfExternal)
                                .filterMap(e -> userManager.getUserOrMaillistByEmail(e))
                                .filterMap(u -> u.getEmailIfMaillist().map(ParticipantId::invitationIdForExternalUser));

                        possibleIds = user.getPossibleIds().plus(maillists);
                    } else {
                        possibleIds = user.getPossibleIds();
                    }

                    Function<Event, Option<UserParticipantInfo>> find =
                            e -> possibleIds.iterator().filterMap(id -> eventInvitationManager
                                    .getUserEventSharing(e.getId(), id)).nextO();

                    masterParticipant = event.filter(e -> e.getRecurrenceId().isPresent())
                            .flatMapO(eventRoutines::getMainInstOfEvent).flatMapO(find);

                    participant = find.apply(event.get());
                }

                return new FoundEvent(
                        Option.of(new EventIdLogDataJson(mainEvent.get(), event.get())),
                        participant.map(ParticipantInfo::getDecision),
                        participant.flatMapO(UserParticipantInfo::getPrivateToken),
                        masterParticipant.isPresent(), false);
            }
        }
        boolean isDeleted = eventRoutines.findDeletedMasterExistsByExternalIdAndParticipantEmails(
                mailerEvent.getExternalId(), group.getParticipantEmailsSafe());

        return new FoundEvent(Option.empty(), Option.empty(), Option.empty(), false, isDeleted);
    }
}
