package ru.yandex.calendar.logic.mailer.change;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.beans.generated.MailerEvent;
import ru.yandex.calendar.logic.mailer.model.MailerParticipant;
import ru.yandex.calendar.logic.sending.param.EventMessageChanged;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;

/**
 * @author dbrylev
 */
public class MailerEventChange {

    private final Option<MailerEvent> previous;
    private final Option<MailerEvent> current;

    private MailerEventChange(Option<MailerEvent> previous, Option<MailerEvent> current) {
        this.previous = previous;
        this.current = current;
    }

    private static ListF<MailerParticipant> userAttendeesChanges(
            Option<MailerEvent> from, Option<MailerEvent> to, boolean common)
    {
        SetF<ParticipantId> fromIds = from.flatMap(
                e -> e.getAttendees().attendees.map(MailerParticipant::getId)).unique();

        return to.flatMap(e -> e.getAttendees().attendees.filter(
                a -> a.getId().isAnyUser() && (common == fromIds.containsTs(a.getId()))));
    }

    public ListF<MailerParticipant> getNewUserAttendees() {
        return previous.isPresent() ? userAttendeesChanges(previous, current, false) : Cf.list();
    }

    public ListF<MailerParticipant> getCurrentUserAttendees() {
        return userAttendeesChanges(previous.orElse(current), current, true);
    }

    public ListF<MailerParticipant> getRemovedUserAttendees() {
        return userAttendeesChanges(current, previous, false);
    }

    public static ListF<Long> getResourceIds(MailerEvent event) {
        return event.getAttendees().attendees.plus(event.getOrganizer())
                .filterMap(p -> p.getId().getResourceIdIfResource());
    }

    public static boolean hasResources(MailerEvent event) {
        return event.getOrganizer().exists(o -> o.getId().isResource())
                || event.getAttendees().attendees.exists(o -> o.getId().isResource());
    }

    public EventMessageChanged findChanged() {
        Function1B<Function<MailerEvent, ?>> changed = f ->
                current.exists(c -> previous.exists(p -> !f.apply(p).equals(f.apply(c))));

        ListF<EventMessageChanged.Field> changes = Cf.arrayList();

        if (changed.apply(MailerEvent::getName)) {
            changes.add(EventMessageChanged.Field.NAME);
        }
        if (changed.apply(MailerEvent::getDescription)) {
            changes.add(EventMessageChanged.Field.DESCRIPTION);
        }
        if (changed.apply(e -> getResourceIds(e).unique())
            || current.exists(MailerEventChange::hasResources) && changed.apply(MailerEvent::getLocation))
        {
            changes.add(EventMessageChanged.Field.LOCATION);
        }
        if (changed.apply(e -> e.getOrganizer().map(MailerParticipant::getId))) {
            changes.add(EventMessageChanged.Field.ORGANIZER);
        }
        if (getNewUserAttendees().isNotEmpty() || getRemovedUserAttendees().isNotEmpty()) {
            changes.add(EventMessageChanged.Field.GUESTS);
        }

        if (changed.apply(MailerEvent::getIsAllDay) || changed.apply(e -> new Duration(e.getStartTs(), e.getEndTs()))) {
            changes.add(EventMessageChanged.Field.TIME);

        } else if (changed.apply(MailerEvent::getExternalId)) {
            if (changed.apply(MailerEvent::getRepetition)) {
                changes.add(EventMessageChanged.Field.TIME);
            }
        } else if (changed.apply(MailerEvent::getRecurrenceId)) {
            if (!current.get().getRecurrenceId().isSome(current.get().getStartTs())) {
                changes.add(EventMessageChanged.Field.TIME);
            }
        } else {
            if (changed.apply(MailerEvent::getStartTs) || changed.apply(MailerEvent::getRepetition)) {
                changes.add(EventMessageChanged.Field.TIME);
            }
        }
        return new EventMessageChanged(changes);
    }

    public MailerEvent getCurrentOrPrevious() {
        return current.orElse(previous).getOrThrow("None is defined");
    }

    public Option<MailerEvent> getCurrent() {
        return current;
    }

    public Option<MailerEvent> getPrevious() {
        return previous;
    }

    public boolean isIgnored() {
        return !previous.isPresent() && !current.isPresent();
    }

    public boolean isUpdated() {
        return previous.isPresent() && current.isPresent();
    }

    public boolean isCreated() {
        return !previous.isPresent() && current.isPresent();
    }

    public boolean isCancelled() {
        return previous.isPresent() && !current.isPresent();
    }

    public static MailerEventChange ignored() {
        return new MailerEventChange(Option.empty(), Option.empty());
    }

    public static MailerEventChange updated(MailerEvent previous, MailerEvent current) {
        return new MailerEventChange(Option.of(previous), Option.of(current));
    }

    public static MailerEventChange created(MailerEvent event) {
        return new MailerEventChange(Option.empty(), Option.of(event));
    }

    public static MailerEventChange cancelled(MailerEvent event) {
        return new MailerEventChange(Option.of(event), Option.empty());
    }

    public MailerEventChange withCurrent(Option<MailerEvent> event) {
        return new MailerEventChange(previous, event);
    }

    public MailerEventChange withPrevious(Option<MailerEvent> event) {
        return new MailerEventChange(event, current);
    }
}
