package ru.yandex.calendar.logic.mailer.logbroker;

import java.util.Optional;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.function.Function;
import ru.yandex.inside.logbroker.util.TskvUtils;
import ru.yandex.inside.passport.PassportUid;

@Slf4j
public class MailAttachLogEvent {
    public final String tskvFormat;
    public final Instant timestamp;

    public final PassportUid uid;
    public final long mid;

    public final int fidSpecType;

    public final String fileType;
    public final String stid;

    public MailAttachLogEvent(
            String tskvFormat, Instant timestamp,
            PassportUid uid, long mid,
            int fidSpecType, String fileType, String stid)
    {
        this.tskvFormat = tskvFormat;
        this.timestamp = timestamp;
        this.uid = uid;
        this.mid = mid;
        this.fidSpecType = fidSpecType;
        this.fileType = fileType;
        this.stid = stid;
    }

    public static Optional<MailAttachLogEvent> parse(String line) {
        MapF<String, String> data = TskvUtils.extractTskv(line);

        Function<String, String> getString = key -> data.getOrThrow(key, "Missing " + key);

        Function<String, Long> getLong = key -> {
            String value = getString.apply(key);
            return Cf.Long.parseSafe(value).getOrThrow("Invalid " + key + ": ", value);
        };

        val mid = getString.apply("mid");
        if (mid.isEmpty()) {
            log.warn("Got event with empty 'mid' field; sessionId = {}", getString.apply("sessionId"));
            return Optional.empty();
        }

        return Optional.of(
            new MailAttachLogEvent(
                getString.apply("tskv_format"), new Instant(getLong.apply("unixtime") * 1000),
                PassportUid.cons(getLong.apply("uid")), getLong.apply("mid"),
                getLong.apply("fidSpecType").intValue(), getString.apply("fileType"), getString.apply("stid")
            )
        );
    }

    public MailAttach getAttach() {
        return new MailAttach(uid, mid, stid);
    }
}
