package ru.yandex.calendar.logic.mailer.logbroker;

import java.util.Optional;
import java.util.concurrent.TimeUnit;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.calendar.frontend.worker.task.HandleMailAttachTask;
import ru.yandex.calendar.logic.mailer.MailerUsers;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.salr.logreader.LogListener;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.Instrument;
import ru.yandex.misc.monica.core.blocks.Meter;
import ru.yandex.misc.monica.core.blocks.Statistic;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.time.TimeUtils;

import static java.lang.Math.max;

@Slf4j
public class MailAttachLogListener implements LogListener, MonicaContainer {
    private static final String PARSE_ERROR_METRIC = "application.mail_attach_log_listener.parse_error";
    private static final String SKIPPED_EVENTS_METRIC = "application.mail_attach_log_listener.skipped_events";
    private static final String PROCESSED_EVENTS_METRIC = "application.mail_attach_log_listener.processed_events";
    private static final String EVENT_TIME_LAG_TIMER = "application.mail_attach_log_listener.time.lag";
    public static final int DELETED_FOLDER_ID = 3;
    @Autowired
    public MailerUsers mailerUsers;
    @Autowired
    private BazingaTaskManager bazingaTaskManager;
    @Autowired
    private MeterRegistry registry;

    @MonicaMetric(description = "Parse errors")
    @GroupByDefault
    final Meter parseErrors = new Meter();

    @MonicaMetric(description = "Skipped events")
    @GroupByDefault
    final Meter skippedEvents = new Meter();

    @MonicaMetric(description = "Event processing time lag statistic")
    @GroupByDefault
    final Statistic eventTimeLag = new Statistic();

    @MonicaMetric(description = "Event processing time lead")
    @GroupByDefault
    final Statistic eventTimeLead = new Statistic();

    @MonicaMetric(description = "Event processing")
    @GroupByDefault
    final Instrument eventProcessing = new Instrument();


    public boolean isAccept(MailAttachLogEvent event) {
        return mailerUsers.isMailerUser(event.uid) && event.fileType.equals("text/calendar")
                && (event.fidSpecType == DELETED_FOLDER_ID || mailerUsers.isMailerUserStrict(event.uid));
    }

    public void processEvent(MailAttachLogEvent event) {
        bazingaTaskManager.schedule(new HandleMailAttachTask(event.getAttach()));
    }

    @Override
    public final void processLogLine(String line) {
        Optional<MailAttachLogEvent> optEvent;

        try {
            optEvent = MailAttachLogEvent.parse(line);
        } catch(RuntimeException e) {
            registry.counter(PARSE_ERROR_METRIC).increment();
            parseErrors.inc();
            log.error("Error while parsing line {}: {}", line, e);
            return;
        }

        if (optEvent.isEmpty()) {
            return;
        }

        final var event = optEvent.get();  // lombok.val leads to compile error

        long receiveDelayMs = System.currentTimeMillis() - event.timestamp.getMillis();
        registry.timer(EVENT_TIME_LAG_TIMER).record(max(receiveDelayMs, 0), TimeUnit.MILLISECONDS);

        if (receiveDelayMs >= 0) {
            eventTimeLag.update(receiveDelayMs);
            eventTimeLead.update(0);
        } else {
            eventTimeLag.update(0);
            eventTimeLead.update(Math.abs(receiveDelayMs));
        }

        if (!isAccept(event)) {
            registry.counter(SKIPPED_EVENTS_METRIC).increment();
            skippedEvents.inc();
            return;
        }

        log.info("Event time lag: {}", TimeUtils.millisecondsToSecondsString(receiveDelayMs));
        eventProcessing.measure(() -> processEvent(event));
        registry.counter(PROCESSED_EVENTS_METRIC).increment();
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName("mail-attach-log",
                new MetricName("mail-attach-log", "processing"), "Mail attach log processing");
    }
}
