package ru.yandex.calendar.logic.mailer.model;

import lombok.extern.slf4j.Slf4j;
import net.fortuna.ical4j.model.component.VEvent;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.beans.generated.MailerEvent;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.ics.IcsUtils;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsVEventGroup;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsVTimeZones;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsMethod;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

/**
 * @author dbrylev
 */
@Slf4j
public class MailerGroup {
    private final ListF<MailerEvent> events;

    public MailerGroup(ListF<MailerEvent> events) {
        Validate.notEmpty(events);
        this.events = events;
    }

    public MailerGroupId getId() {
        return new MailerGroupId(events.first().getUid(), events.first().getExternalId());
    }

    public ListF<MailerEvent> getEvents() {
        return events;
    }

    public Option<MailerEvent> getMaster() {
        return events.filter(e -> !e.getRecurrenceId().isPresent()).singleO();
    }

    public ListF<MailerEvent> getRecurrences() {
        return events.filter(e -> e.getRecurrenceId().isPresent());
    }

    public MailerEvent getMasterOrFirstEvent() {
        return getMaster().getOrElse(() -> getRecurrences().first());
    }

    public static MailerGroup fromIcs(
            PassportUid uid, IcsCalendar calendar, IcsVTimeZones tzs,
            Function<ListF<Email>, Tuple2List<Email, ParticipantId>> participantIdsResolver, Instant dtstampFallback)
    {
        IcsVEventGroup group = calendar.getEventsGroupedByUidWithExpiredFiltered().single();

        MapF<Email, ParticipantId> participantIds = participantIdsResolver.apply(group.getParticipantEmailsSafe()).toMap();

        return new MailerGroup(group.getEvents().map(vevent -> {
            MailerEvent event = new MailerEvent();
            VEvent component = vevent.toComponent();

            event.setUid(uid);
            event.setExternalId(group.getUid().getOrThrow("UID"));

            event.setRecurrenceId(vevent.getRecurrenceIdInstant(tzs));
            event.setInstanceId(event.getRecurrenceId().map(Instant::getMillis).getOrElse(0L));

            event.setStartTs(vevent.getStart(tzs));
            event.setEndTs(vevent.getEnd(tzs));
            event.setIsAllDay(vevent.isAllDay());

            event.setRepetition(vevent.getRRules().<Option<MailerRepetition>>foldLeft(Option.empty(),
                    (agg, rr) -> agg.orElse(() -> MailerRepetition.fromIcs(rr, vevent.getStart().getDateTime(tzs)))));
            event.setTimezoneId(vevent.getStart().getDateTime(tzs).getZone().getID());

            event.setName(IcsUtils.getNotBlankValue(component.getSummary()).getOrElse(EventRoutines.DEFAULT_NAME));

            if (calendar.getProdId().exists(pid -> pid.contains("Exchange"))
                && !calendar.getMethod().sameMethodAs(IcsMethod.REQUEST))
            {
                event.setName(StringUtils.substringAfter(event.getName(), ":").trim());
            }

            event.setLocation(IcsUtils.getNotBlankValue(component.getLocation()));

            event.setDescription(IcsUtils.getNotBlankValue(component.getDescription()));
            event.setUrl(IcsUtils.getNotBlankValue(component.getUrl()));

            event.setOrganizer(vevent.getOrganizer()
                    .flatMapO(o -> MailerParticipant.fromIcs(o, participantIds::getOrThrow)));

            event.setAttendees(new MailerAttendees(vevent.getAttendees()
                    .filterMap(a -> MailerParticipant.fromIcs(a, participantIds::getOrThrow))
                    .filterNot(a -> event.getOrganizer().exists(o -> o.getId().equals(a.getId())))));

            event.setSequence(vevent.getSequence().getOrElse(0));
            event.setDtstamp(vevent.getDtStampInstant(tzs).getOrElse(dtstampFallback));

            return event;
        }));
    }
}
