package ru.yandex.calendar.logic.mailer.model;

import java.text.ParseException;

import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MailDateFormat;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.contact.UnivContact;
import ru.yandex.calendar.logic.ics.IcsUtils;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsProdId;
import ru.yandex.calendar.logic.mailer.logbroker.MailAttach;
import ru.yandex.calendar.logic.sending.param.MessageDestination;
import ru.yandex.calendar.logic.sending.param.MessageOverrides;
import ru.yandex.calendar.logic.user.UserOrMaillist;
import ru.yandex.commune.mail.BodyPart;
import ru.yandex.commune.mail.DefaultContent;
import ru.yandex.commune.mail.HeaderNames;
import ru.yandex.commune.mail.MailAddress;
import ru.yandex.commune.mail.MailMessage;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author dbrylev
 */
public class MailerMail {
    public final Instant date;
    public final String subject;
    public final String messageId;
    public final MailAttach attach;
    public final UnivContact sender;
    public final IcsCalendar ics;

    public MailerMail(
            Instant date, String subject, String messageId, MailAttach attach,
            UnivContact sender, IcsCalendar ics)
    {
        this.date = date;
        this.subject = subject;
        this.messageId = messageId;
        this.attach = attach;
        this.sender = sender;
        this.ics = ics;
    }

    public MailerMail forSending() {
        return new MailerMail(date, subject,
                transformMessageId(messageId), attach, sender,
                ics.withProperty(new IcsProdId("-//Yandex Calendar//" + ics.getProdId().mkString(""), Cf.list())));
    }

    public MessageOverrides toOverrides(UserOrMaillist user) {
        return new MessageOverrides(
                Option.of(date), Option.of(messageId), Option.empty(),
                Option.of(user.isMaillist()
                        ? MessageDestination.MAILLIST_SHARED_FOLDER
                        : MessageDestination.NOT_FOR_EXCHANGE));
    }

    public static String transformMessageId(String messageId) {
        return StringUtils.removeEnd(StringUtils.removeStart(messageId, "<"), ">").replace('@', '_') + "@calendar.yandex";
    }

    private static final MailDateFormat mailDateFormat = new MailDateFormat();

    public static Either<MailerMail, String> parseOrRefusal(MailMessage message, MailAttach attach) {
        Instant date;
        try {
            date = new Instant(mailDateFormat.parse(message.getHeader(HeaderNames.DATE).getOrThrow("Missed")));

        } catch (ParseException | RuntimeException e) {
            return Either.right("Date parsing error: " + ExceptionUtils.getAllMessages(e));
        }

        Option<String> subject = message.getSubject();

        if (!subject.isPresent()) {
            return Either.right("Missing subject");
        }

        Option<String> messageId = message.getMessageId();

        if (!messageId.isPresent()) {
            return Either.right("Missing message id");
        }

        MailAddress sender;
        try {
            InternetAddress address = new InternetAddress(message.getHeader(HeaderNames.FROM).getOrThrow("Missed"));
            sender = new MailAddress(new Email(address.getAddress()), address.getPersonal());

        } catch (AddressException | RuntimeException e) {
            return Either.right("Sender parsing error: " + ExceptionUtils.getAllMessages(e));
        }

        IcsCalendar ics;
        try {
            BodyPart icsPart = message.getBottomPartsWithTypeSubtype("text/calendar").firstO().getOrThrow("Missed");
            Object value = ((DefaultContent) icsPart.getContent()).readValue();

            byte[] bytes = value instanceof String ? ((String) value).getBytes() : (byte[]) value;

            ics = icsPart.getContentType().get().getCharsetName()
                    .map(charset -> IcsUtils.parseBytes(bytes, charset))
                    .getOrElse(() -> IcsUtils.parseBytesInGuessedEncoding(bytes));

        } catch (RuntimeException e) {
            return Either.right("Ics parsing error: " + ExceptionUtils.getAllMessages(e));
        }

        return Either.left(new MailerMail(
                date, subject.get(), messageId.get(), attach,
                new UnivContact(sender.getEmail(), sender.getPersonal(), Option.empty()), ics));
    }
}
