package ru.yandex.calendar.logic.mailer.model;

import net.fortuna.ical4j.model.parameter.PartStat;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.beans.DatabaseBender;
import ru.yandex.calendar.logic.beans.JsonizableValue;
import ru.yandex.calendar.logic.ics.iv5j.ical.parameter.IcsPartStat;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsEmailPropertyBase;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsOrganizer;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.participant.ExternalUserParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.ResourceParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.YandexUserParticipantInfo;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.annotation.FillWithSomething;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderParseSubclasses;
import ru.yandex.misc.bender.annotation.SubclassTypeDefinition;
import ru.yandex.misc.email.Email;

/**
 * @author dbrylev
 */
@BenderBindAllFields
@BenderParseSubclasses(jsonFieldName = "type", typeDefinition = SubclassTypeDefinition.FIELD, value = {
        MailerParticipant.User.class,
        MailerParticipant.Resource.class,
        MailerParticipant.External.class,
})
public abstract class MailerParticipant implements JsonizableValue {

    public static final BenderParserSerializer<MailerParticipant> bender =
            DatabaseBender.mapper.createParserSerializer(MailerParticipant.class);

    public final Email email;
    public final Decision decision;

    public MailerParticipant(Email email, Decision decision) {
        this.email = email;
        this.decision = decision;
    }

    public abstract ParticipantId getId();

    public Email getEmail() {
        return email;
    }

    public Option<String> getNameIfExternal() {
        return Option.when(this instanceof External, () -> ((External) this).name);
    }

    public Option<Email> getEmailIfExternal() {
        return Option.when(this instanceof External, () -> ((External) this).email);
    }

    public static Option<MailerParticipant> fromIcs(
            IcsEmailPropertyBase prop, Function<Email, ParticipantId> idResolver)
    {
        return Option.when(prop.isEmail(), () -> {
            ParticipantId id = idResolver.apply(prop.getEmail());

            Decision decision = prop.<IcsPartStat>getParameter(PartStat.PARTSTAT)
                    .flatMapO(Decision::findByIcsPartStat)
                    .getOrElse(prop instanceof IcsOrganizer ? Decision.YES : Decision.UNDECIDED);

            if (id.isYandexUser()) {
                return new MailerParticipant.User(id.getUid(), prop.getEmail(), decision);
            } else if (id.isResource()) {
                return new MailerParticipant.Resource(id.getResourceId(), prop.getEmail(), decision);
            } else {
                return new MailerParticipant.External(id.getInviteeEmailForExternalUser(), prop.getCn(), decision);
            }
        });
    }

    public static MailerParticipant fromParticipant(ParticipantInfo participant) {
        if (participant instanceof YandexUserParticipantInfo) {
            YandexUserParticipantInfo p = (YandexUserParticipantInfo) participant;
            return new MailerParticipant.User(p.getUidSome(), p.getEmail(), p.getDecision());
        }
        if (participant instanceof ResourceParticipantInfo) {
            ResourceParticipantInfo p = (ResourceParticipantInfo) participant;
            return new MailerParticipant.Resource(p.getResourceId(), p.getEmail(), p.getDecision());
        }
        if (participant instanceof ExternalUserParticipantInfo) {
            ExternalUserParticipantInfo p = (ExternalUserParticipantInfo) participant;
            return new MailerParticipant.External(p.getEmail(), p.getName(), p.getDecision());
        }
        throw new IllegalArgumentException("Unexpected participant type: " + participant.getClass());
    }

    public static MailerParticipant fromDbValue(String value) {
        return bender.getParser().parseJson(value);
    }

    @Override
    public String toDbValue() {
        return new String(bender.getSerializer().serializeJson(this));
    }

    @Override
    public String toString() {
        return toDbValue();
    }

    @BenderBindAllFields
    public static class User extends MailerParticipant {
        public final PassportUid uid;

        @FillWithSomething
        public User(PassportUid uid, Email email, Decision decision) {
            super(email, decision);
            this.uid = uid;
        }

        @Override
        public ParticipantId getId() {
            return ParticipantId.yandexUid(uid);
        }
    }

    @BenderBindAllFields
    public static class Resource extends MailerParticipant {
        public final long id;

        @FillWithSomething
        public Resource(long id, Email email, Decision decision) {
            super(email, decision);
            this.id = id;
        }

        @Override
        public ParticipantId getId() {
            return ParticipantId.resourceId(id);
        }
    }

    @BenderBindAllFields
    public static class External extends MailerParticipant {
        public final String name;

        @FillWithSomething
        public External(Email email, String name, Decision decision) {
            super(email, decision);
            this.name = name;
        }

        @Override
        public ParticipantId getId() {
            return ParticipantId.invitationIdForExternalUser(email);
        }
    }
}
