package ru.yandex.calendar.logic.mailer.model;

import org.joda.time.DateTime;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.DatabaseBender;
import ru.yandex.calendar.logic.beans.JsonizableValue;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.RepetitionFields;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.repetition.RegularRepetitionRule;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsRRule;
import ru.yandex.misc.annotation.FillWithSomething;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author dbrylev
 */
@BenderBindAllFields
public class MailerRepetition extends DefaultObject implements JsonizableValue {

    public static final BenderParserSerializer<MailerRepetition> bender =
            DatabaseBender.mapper.createParserSerializer(MailerRepetition.class);

    public final RegularRepetitionRule type;
    public final int each;
    public final Option<LocalDate> due;
    @BenderPart(wrapperName = "weeklyDays")
    public final Option<ListF<String>> weeklyDays;
    public final Option<Boolean> monthlyLastweek;

    @FillWithSomething
    public MailerRepetition(
            RegularRepetitionRule type, int each, Option<LocalDate> due,
            Option<ListF<String>> weeklyDays, Option<Boolean> monthlyLastweek)
    {
        this.type = type;
        this.each = each;
        this.due = due;
        this.weeklyDays = weeklyDays;
        this.monthlyLastweek = monthlyLastweek;
    }

    public static Option<MailerRepetition> fromIcs(IcsRRule rr, DateTime startDt) {
        return RepetitionUtils.getRepetitionSafe(rr.getRecur(), startDt.toInstant(), startDt.getZone())
                .map(r -> new MailerRepetition(
                        r.getType(), r.getREach().get(),
                        r.getDueTs().map(d -> EventRoutines.convertDueTsToUntilDate(d, startDt.getZone())),
                        r.getFieldValueO(RepetitionFields.R_WEEKLY_DAYS).map(s -> Cf.x(s.split(","))),
                        r.getFieldValueO(RepetitionFields.R_MONTHLY_LASTWEEK)));
    }

    public Repetition toRepetition(DateTime start) {
        Repetition rep = new Repetition();
        rep.setType(type);
        rep.setREach(each);
        rep.setDueTs(due.map(d -> EventRoutines.calculateDueTsFromUntilDate(start.toInstant(), d, start.getZone())));
        rep.setRWeeklyDays(weeklyDays.map(days -> days.mkString(",")));
        rep.setRMonthlyLastweek(monthlyLastweek);

        return rep;
    }

    public static MailerRepetition fromDbValue(String value) {
        return bender.getParser().parseJson(value);
    }

    @Override
    public String toDbValue() {
        return new String(bender.getSerializer().serializeJson(this));
    }

    @Override
    public String toString() {
        return toDbValue();
    }
}
