package ru.yandex.calendar.logic.notification;

import org.jdom.Element;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.inside.passport.PassportSid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.digest.Md5;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.CamelWords;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.TimeUtils;
import ru.yandex.misc.xml.jdom.JdomUtils;

/**
 * Notifications, signed with control-data, for/from services which support it.
 * [TAG: BALANCE_GENERALIZE]: {@link #SALT} secret is shared for now for all sid-s.
 * @author ssytnik
 */
public class ControlDataNotification {
    private static final Logger logger = LoggerFactory.getLogger(ControlDataNotification.class);

    private static final String SALT = "26b9cfce4e1349c8bb24f54f506988d7";
    private static final String RESP_TAG_NAME = "result";
    private static final String RESP_VALUE_OK = "ok";
    private static final String RESP_VALUE_TRY_AGAIN = "try-again";
    private static final String SEND_PREFIX = "CDN.send(): ";

    @Value("${balance_notification.url}")
    private String balanceNotificationUrlBase;

    @Autowired
    EventDao eventDao;

    /**
     * Notification type for service calls made by calendar. Is passed to the
     * calling url. If {@link #NOTIFY} fails, an another attempt may be tried.
     * @author ssytnik
     */
    public static enum CalendarNotificationType {
        CREATE, NOTIFY, DELETE;
        public String toLowerCase() { return CamelWords.parse(this.name()).toDbName(); }
    }
    // NOTE: there is no need for notification type for 'service -> calendar' calls
    // because we do not use them to evaluate control data

    public static boolean isSupportedBy(PassportSid sid) {
        // [TAG: BALANCE_GENERALIZE]: may be substituted with AuxBase.in(sid, ...)
        return sid.sameAs(PassportSid.BALANCE);
    }

    protected static String evalControlData(PassportUid uid, String eventExtId, String... optParams) {
        return Md5.A.digest(Cf.list(SALT, uid.toString(), eventExtId).plus(optParams).mkString(":")).hex();
    }

    protected static String getFullNotificationUrl(
            String notificationUrlBase, PassportUid uid, String eventExtId, String nowTsStr,
            CalendarNotificationType notificationType, String controlData)
    {
        return String.format(
            "%s?ntf_type=%s&uid=%d&external_id=%s&ntf_ts=%s&control_data=%s",
            notificationUrlBase, UrlUtils.urlEncode(notificationType.toLowerCase()), uid.getUid(),
            UrlUtils.urlEncode(eventExtId), UrlUtils.urlEncode(nowTsStr), UrlUtils.urlEncode(controlData)
        );
    }

    /**
     * Calls for service, telling about event creation/notification time/deletion.
     * @param uid not necessarily event creator uid - service checks this itself
     * @return For {@link CalendarNotificationType#NOTIFY}, can return error status.
     */
    public Tuple2<NotificationStatus, Option<String>> send(
            Event event, PassportUid uid, String eventExtId, Instant nowTs, CalendarNotificationType notificationType)
    {
        PassportSid sid = event.getSid();
        Validate.isTrue(isSupportedBy(sid));
        Tuple2<NotificationStatus, Option<String>> res = Tuple2.tuple(NotificationStatus.PROCESSED, Option.<String>empty());
        // [TAG: BALANCE_GENERALIZE] for now, we use Moscow timezone.
        // For some services, user (dtf's) timezone may be used.
        String nowTsStr = DateTimeFormatter.formatForMachines(nowTs, TimeUtils.EUROPE_MOSCOW_TIME_ZONE);
        String controlData = evalControlData(uid, eventExtId, nowTsStr, notificationType.toLowerCase());
        String fullNtfUrl = getFullNotificationUrl(
            balanceNotificationUrlBase, uid, eventExtId, nowTsStr, notificationType, controlData
        );
        //logger.debug("Full url = " + fullNtfUrl);
        try {
            String response = ApacheHttpClientUtils.downloadString(fullNtfUrl, Timeout.seconds(5));
            Element eXml = JdomUtils.I.readRootElement(response);
            String eXmlName = eXml.getName(), eXmlText = eXml.getText();
            if (!eXmlName.equals(RESP_TAG_NAME) || !Cf.list(RESP_VALUE_OK, RESP_VALUE_TRY_AGAIN).containsTs(eXmlText)) {
                String mes = SEND_PREFIX + "wrong service response: tag = '" + eXmlName + "', text = '" + eXmlText + "'";
                res = Tuple2.tuple(NotificationStatus.FATAL_ERROR, Option.of(mes));
            }
        } catch(CommandRunException cre) {
            logger.error((SEND_PREFIX + "error caught after attempting AuxInet.getUrlContents()"));
            if (cre.isSituation(Situation.CANNOT_OPEN_URL_CONN)) {
                String mes = SEND_PREFIX + "could not open url connection";
                res = Tuple2.tuple(NotificationStatus.TRY_AGAIN, Option.of(mes));
            } else {
                throw cre;
            }
        }
        if (CalendarNotificationType.NOTIFY != notificationType && NotificationStatus.PROCESSED != res.get1()) {
            throw new CommandRunException(res.get2().getOrNull());
        }
        return res;
    }

    private static boolean isValid(PassportSid sid, PassportUid uid, String eventExtId, String controlData) {
        if (!isSupportedBy(sid)) {
            String msg = "Sid: " + sid + " is not allowed to call calendar control-data methods";
            throw new CommandRunException(msg);
        }
        return evalControlData(uid, eventExtId).equals(controlData);
    }

    public static void ensureValid(PassportSid sid, PassportUid uid, String eventExtId, String controlData) {
        if (!isValid(sid, uid, eventExtId, controlData)) {
            String msg = "Invalid: uid = " + uid + ", eventExtId: " + eventExtId + ", controlData: " + controlData;
            throw new CommandRunException(msg);
        }
    }
}
