package ru.yandex.calendar.logic.notification;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.IncorrectResultSizeDataAccessException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.logic.beans.GenericBeanDao;
import ru.yandex.calendar.logic.beans.generated.EventNotification;
import ru.yandex.calendar.logic.beans.generated.EventNotificationFields;
import ru.yandex.calendar.logic.beans.generated.EventNotificationHelper;
import ru.yandex.calendar.logic.beans.generated.LayerFields;
import ru.yandex.calendar.logic.beans.generated.LayerNotification;
import ru.yandex.calendar.logic.beans.generated.LayerNotificationFields;
import ru.yandex.calendar.logic.beans.generated.LayerNotificationHelper;
import ru.yandex.calendar.util.db.BeanRowMapper;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlQueryUtils;

/**
 * @author Daniel Brylev
 */
public class NotificationDao extends CalendarJdbcDaoSupport {

    @Autowired
    private GenericBeanDao genericBeanDao;

    public ListF<Long> saveEventNotificationsBatch(ListF<EventNotification> notifications) {
        return genericBeanDao.insertBeansBatchGetGeneratedKeys(notifications);
    }

    public ListF<Long> saveLayerNotificationsBatch(ListF<LayerNotification> notifications) {
        return genericBeanDao.insertBeansBatchGetGeneratedKeys(notifications);
    }

    @RunWithRandomTest(possible = IncorrectResultSizeDataAccessException.class)
    public Tuple2List<Long, ListF<EventNotification>> findEventNotificationsByEventUserIdsSafe(ListF<Long> eventUserIds) {
        String sql = "SELECT * FROM event_notification WHERE event_user_id " + SqlQueryUtils.inSet(eventUserIds);

        if (skipQuery(eventUserIds, sql)) return Tuple2List.tuple2List();

        Comparator<EventNotification> c =
                (Comparator<EventNotification>) Comparator.<EventNotification>constEqualComparator()
                        .thenComparing(EventNotification::getOffsetMinute)
                        .thenComparing(EventNotification::getChannel);

        return getJdbcTemplate().queryForList(sql, EventNotification.class)
                .zipWith(EventNotification.getEventUserIdF()).groupBy2().entries().map2(l -> l.sorted(c));
    }

    @RunWithRandomTest(possible = IncorrectResultSizeDataAccessException.class)
    public Tuple2List<Long, ListF<LayerNotification>> findNotificationsByLayerUserIdsSafe(ListF<Long> layerUserIds) {
        String sql = "SELECT * FROM layer_notification WHERE layer_user_id " + SqlQueryUtils.inSet(layerUserIds);

        if (skipQuery(layerUserIds, sql)) return Tuple2List.tuple2List();

        Comparator<LayerNotification> c =
                (Comparator<LayerNotification>) Comparator.<LayerNotification>constEqualComparator()
                        .thenComparing(LayerNotification::getOffsetMinute)
                        .thenComparing(LayerNotification::getChannel);

        return getJdbcTemplate().queryForList(sql, LayerNotification.class)
                .zipWith(LayerNotification.getLayerUserIdF()).groupBy2().entries().map2(l -> l.sorted(c));
    }

    @RunWithRandomTest
    public MapF<Long, ListF<LayerNotification>> findLayerCreatorNotificationsByLayerIds(CollectionF<Long> layerIds) {
        SqlCondition lc = LayerFields.ID.column().inSet(layerIds);
        BeanRowMapper<LayerNotification> lnRm = LayerNotificationHelper.INSTANCE.offsetRowMapper(1);

        String q = "SELECT l.id, " + lnRm.columns("ln.") + " FROM layer_notification ln, layer_user lu, layer l"
                + " WHERE ln.layer_user_id = lu.id AND lu.layer_id = l.id AND lu.uid = l.creator_uid"
                + lc.andSqlForTable("l");

        if (skipQuery(lc, q, lc.args())) return Cf.map();

        return getJdbcTemplate().query2(q, (rs, rn) -> rs.getLong(1), lnRm, lc.args()).groupBy1();

    }

    @RunWithRandomTest
    public ListF<EventNotification> findEventNotificationsSortedByNextSendTs(
            SqlCondition notificationCondition, SqlCondition eventUserCondition)
    {
        String sql = "SELECT en.* FROM event_notification en" +
                " INNER JOIN event_user eu ON eu.id = en.event_user_id" +
                " WHERE (" + eventUserCondition.sqlForTable("eu") + ")" +
                " AND (" + notificationCondition.sqlForTable("en") + ")" +
                " ORDER BY en.next_send_ts";
        ListF<?> sqlArgs = Cf.list(eventUserCondition.args(), notificationCondition.args());

        return getJdbcTemplate().queryForList(sql, EventNotification.class, sqlArgs);
    }

    public ListF<EventNotification> findEventNotificationsByEventUserIdsAndChannel(ListF<Long> ids, Channel channel) {
        return findEventNotifications(EventNotificationFields.EVENT_USER_ID.column().inSet(ids)
                .and(EventNotificationFields.CHANNEL.eq(channel)));
    }

    public Option<EventNotification> findEventNotificationById(long id) {
        return findEventNotifications(EventNotificationFields.ID.eq(id)).singleO();
    }

    private ListF<EventNotification> findEventNotifications(SqlCondition condition) {
        return genericBeanDao.loadBeans(EventNotificationHelper.INSTANCE, condition);
    }

    public void findEventNotificationsSortedByNextSendTs(
            SqlCondition notificationCondition,
            int rowsPerCallback, Function<ListF<EventNotification>, ListF<Long>> callback)
    {
        genericBeanDao.loadBeans(
                EventNotificationHelper.INSTANCE, notificationCondition,
                EventNotificationFields.NEXT_SEND_TS, rowsPerCallback, callback);
    }

    @RunWithRandomTest
    public void updateEventNotificationNextSendTsById(long notificationId, Option<Instant> nextSendTs) {
        String sql = "UPDATE event_notification SET next_send_ts = ? WHERE id = ?";
        getJdbcTemplate().update(sql, nextSendTs.getOrNull(), notificationId);
    }

    @RunWithRandomTest
    public void updateEventNotificationNextSendTsBatch(ListF<EventNotification> notifications) {
        genericBeanDao.batchUpdateBeansFieldsByIds(notifications,
                Cf.<MapField<?>>list(EventNotificationFields.NEXT_SEND_TS));
    }

    @RunWithRandomTest
    public void deleteEventNotificationsByEventUserIds(ListF<Long> eventUserIds) {
        String sql = "DELETE FROM event_notification WHERE event_user_id " + SqlQueryUtils.inSet(eventUserIds);

        if (skipQuery(eventUserIds, sql)) return;

        getJdbcTemplate().update(sql);
    }

    @RunWithRandomTest
    public void deleteLayerNotificationsByLayerUserIds(ListF<Long> layerUserIds) {
        String sql = "DELETE FROM layer_notification WHERE layer_user_id " + SqlQueryUtils.inSet(layerUserIds);

        if (skipQuery(layerUserIds, sql)) return;

        getJdbcTemplate().update(sql);
    }

    @RunWithRandomTest
    public void deleteEventNotificationsByIds(ListF<Long> eventNotificationIds) {
        String sql = "DELETE FROM event_notification WHERE id " + SqlQueryUtils.inSet(eventNotificationIds);

        if (skipQuery(eventNotificationIds, sql)) return;

        getJdbcTemplate().update(sql);
    }

    @RunWithRandomTest
    public void deleteEventNotificationsByEventUserUid(PassportUid uid) {
        String sql = "DELETE FROM event_notification en" +
                " USING event_user eu" +
                " WHERE en.event_user_id = eu.id AND eu.uid = ?";
        getJdbcTemplate().update(sql, uid);
    }

    @RunWithRandomTest
    public void deleteLayerNotificationsByLayerUserUid(PassportUid uid) {
        String sql = "DELETE FROM layer_notification ln" +
                " USING layer_user lu" +
                " WHERE ln.layer_user_id = lu.id AND lu.uid = ?";
        getJdbcTemplate().update(sql, uid);
    }

    @RunWithRandomTest
    public void deleteLayerNotificationsByLayerUserIdAndChannels(long layerUserId, SetF<Channel> channels) {
        SqlCondition idCondition = LayerNotificationFields.LAYER_USER_ID.eq(layerUserId);
        SqlCondition inChannelsCondition = LayerNotificationFields.CHANNEL.column().inSet(channels);
        SqlCondition condition = SqlCondition.all(idCondition, inChannelsCondition);

        String sql = "DELETE FROM layer_notification " + condition.whereSql();

        if (skipQuery(condition, sql, condition.args())) return;

        getJdbcTemplate().update(sql, condition.args());
    }

}
