package ru.yandex.calendar.logic.notification;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.logic.beans.generated.EventNotification;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.LayerNotification;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.inside.passport.PassportUid;

/**
 * @author Daniel Brylev
 */
public class NotificationDbManager {

    @Autowired
    private NotificationDao notificationDao;
    @Autowired
    private NotificationRoutines notificationRoutines;
    @Autowired
    private EventUserDao eventUserDao;

    public EventNotifications getNotificationsByEventUserId(long eventUserId) {
        return getNotificationsByEventUserIds(Cf.list(eventUserId)).single();
    }

    public ListF<EventNotifications> getNotificationsByEventUserIds(ListF<Long> eventUserIds) {
        MapF<Long, ListF<EventNotification>> x =
                notificationDao.findEventNotificationsByEventUserIdsSafe(eventUserIds).toMap();
        return eventUserIds
                .zipWith(id -> x.getOrElse(id, Cf.list()))
                .map(EventNotifications::new);
    }

    public Option<EventUserWithNotifications> getEventUserWithNotificationsByUidAndEventId(
            PassportUid uid, long eventId)
    {
        return getEventUsersWithNotificationsByUidAndEventIds(uid, Cf.list(eventId)).singleO();
    }

    public ListF<EventUserWithNotifications> getEventUsersWithNotificationsByUidAndEventIds(
            PassportUid uid, ListF<Long> eventIds)
    {
        return getEventUserWithNotificationsByEventUsers(eventUserDao.findEventUsersByEventIdsAndUid(eventIds, uid));
    }

    public ListF<EventUserWithNotifications> getEventUsersWithNotificationsByEventIds(ListF<Long> eventIds) {
        return getEventUserWithNotificationsByEventUsers(eventUserDao.findEventUsersByEventIds(eventIds));
    }

    public ListF<EventUserWithNotifications> getEventUserWithNotificationsByEventUsers(ListF<EventUser> eventUsers) {
        MapF<Long, EventNotifications> x =
                getNotificationsByEventUserIds(eventUsers.map(EventUser.getIdF()))
                        .toMapMappingToKey(EventNotifications::getEventUserId);
        return eventUsers
                .zipWith(eu -> x.getOrThrow(eu.getId()))
                .map(EventUserWithNotifications.consF());
    }

    public ListF<Notification> getNotificationsByLayerUserId(long layerUserId) {
        return getNotificationsByLayerUserIds(Cf.list(layerUserId)).single().get2();
    }

    public ListF<Notification> getNotificationsByUidAndEventId(PassportUid uid, long eventId) {
        Option<EventUser> eventUser = eventUserDao.findEventUserByEventIdAndUid(eventId, uid);
        return eventUser.isPresent() ?
                getNotificationsByEventUserId(eventUser.get().getId()).getNotifications() :
                Cf.<Notification>list();
    }

    public Tuple2List<Long, ListF<Notification>> getNotificationsByLayerUserIds(ListF<Long> layerUserIds) {
        MapF<Long, ListF<LayerNotification>> x =
                notificationDao.findNotificationsByLayerUserIdsSafe(layerUserIds).toMap();

        return layerUserIds.zipWith(id -> x.getOrElse(id, Cf.list()).map(Notification::of));
    }

    public MapF<Long, ListF<Notification>> getLayerCreatorNotificationsByLayerIds(CollectionF<Long> layerIds) {
        MapF<Long, ListF<LayerNotification>> x = notificationDao.findLayerCreatorNotificationsByLayerIds(layerIds);

        return layerIds.toMapMappingToValue(id -> x.getOrElse(id, Cf.list()).map(Notification::of));
    }

    public void saveEventNotifications(long eventUserId, ListF<Notification> notifications) {
        saveEventNotifications(notifications.map(n -> n.toEventNotification(eventUserId)));
    }

    public void saveEventNotifications(ListF<EventNotification> notifications) {
        notificationDao.saveEventNotificationsBatch(notifications);
    }

    public void updateLayerNotifications(long layerUserId, NotificationsData.Update data) {
        SetF<Channel> channelsToUpdate = data.getChannelsToUpdate();
        notificationDao.deleteLayerNotificationsByLayerUserIdAndChannels(layerUserId, channelsToUpdate);
        notificationDao.saveLayerNotificationsBatch(data.getNotifications().map(n -> n.toLayerNotification(layerUserId)));
    }

    public void saveLayerNotifications(long layerUserId, ListF<Notification> notifications) {
        notificationDao.saveLayerNotificationsBatch(notifications.map(n -> n.toLayerNotification(layerUserId)));
    }

    public void updateEventNotifications(EventNotificationChangesInfo changes) {
        updateEventNotifications(Cf.list(changes));
    }

    public void updateEventNotifications(ListF<EventNotificationChangesInfo> changes) {
        changes = changes.filter(EventNotificationChangesInfo.wasChangesF());
        if (changes.isEmpty()) {
            return;
        }
        ListF<Long> deleteIds = changes.flatMap(EventNotificationChangesInfo.getDeleteNotificationIdsF());
        notificationDao.deleteEventNotificationsByIds(deleteIds);
        notificationDao.saveEventNotificationsBatch(
                changes.flatMap(EventNotificationChangesInfo.getCreateNotificationsF()));
    }

    public void deleteNotificationsByEventUserIds(ListF<Long> eventUserIds) {
        notificationDao.deleteEventNotificationsByEventUserIds(eventUserIds);
    }

    public void deleteNotificationsByLayerUserIds(ListF<Long> layerUserIds) {
        notificationDao.deleteLayerNotificationsByLayerUserIds(layerUserIds);
    }

    public void updateAndRecalcEventNotifications(
            long eventUserId, NotificationsData.Update notifications, ActionInfo actionInfo)
    {
        EventNotifications oldNotifications = getNotificationsByEventUserId(eventUserId);
        EventNotificationChangesInfo changes = NotificationChangesFinder
                .changes(oldNotifications, notifications);
        updateEventNotifications(changes);

        notificationRoutines.recalcNextSendTs(eventUserId, actionInfo);
    }

}
