package ru.yandex.calendar.logic.notification;

import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventNotification;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.telemost.TelemostManager;
import ru.yandex.calendar.util.base.AuxColl;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author ssytnik
 */
public class NotificationRoutines {
    private static final Logger logger = LoggerFactory.getLogger(NotificationRoutines.class);

    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private NotificationDao notificationDao;
    @Autowired
    private NotificationDbManager notificationDbManager;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private TelemostManager telemostManager;

    @Value("${notifications.default.email.offset.minutes}")
    private int notificationsDefaultEmailOffsetMinutes;

    public ListF<Notification> getDefaultNotifications() {
        return EnvironmentType.TESTING.isActive()
                ? Cf.list()
                : Cf.list(Notification.email(Duration.standardMinutes(notificationsDefaultEmailOffsetMinutes)));
    }

    public void recalcNextSendTs(long eventUserId, ActionInfo actionInfo) {
        EventUser eventUser = eventUserDao.findEventUserById(eventUserId);
        EventUserWithNotifications pair = notificationDbManager.getEventUserWithNotificationsByEventUsers(
                Cf.list(eventUser)).single();
        EventAndRepetition event = eventDbManager.getEventAndRepetitionById(eventUser.getEventId());

        recalcNextSendTsForEvents(Cf.list(event), Cf.list(pair), actionInfo);
    }

    public void recalcUserNextSendTs(PassportUid uid, ListF<EventAndRepetition> events, ActionInfo actionInfo) {
        ListF<EventUserWithNotifications> pairs = notificationDbManager.getEventUsersWithNotificationsByUidAndEventIds(
                uid, events.map(EventAndRepetition.getEventIdF()));

        recalcNextSendTsForEvents(events, pairs, actionInfo);
    }

    public void recalcAllNextSendTs(long eventId, ActionInfo actionInfo) {
        recalcAllNextSendTs(Cf.list(eventId), actionInfo);
    }

    public void recalcAllNextSendTs(ListF<Long> eventIds, ActionInfo actionInfo) {
        ListF<EventAndRepetition> events = eventDbManager.getEventsAndRepetitionsByEventIds(eventIds);

        events.forEach(event -> telemostManager.rescheduleGeneration(event, actionInfo));

        ListF<EventUserWithNotifications> pairs =
                notificationDbManager.getEventUsersWithNotificationsByEventIds(eventIds);

        recalcNextSendTsForEvents(events, pairs, actionInfo);
    }

    public void recalcNextSendTsForNewRecurrenceOrTail(
            long masterEventId, long newEventId, ActionInfo actionInfo)
    {
        ListF<Long> eventIds = Cf.list(masterEventId, newEventId);
        ListF<EventAndRepetition> events = eventDbManager.getEventsAndRepetitionsByEventIds(eventIds);

        events.forEach(event -> telemostManager.rescheduleGeneration(event, actionInfo));

        ListF<EventUserWithNotifications> pairs = notificationDbManager.getEventUsersWithNotificationsByEventIds(eventIds);

        EventAndRepetition masterEvent = events.find(EventAndRepetition.getEventIdF().andThenEquals(masterEventId)).get();
        EventAndRepetition newEvent = events.find(EventAndRepetition.getEventIdF().andThenEquals(newEventId)).get();

        MapF<PassportUid, DateTimeZone> tzByUid = dateTimeManager.getTimeZonesForUids(
                pairs.map(EventUserWithNotifications.getUidF())).toMap();

        ListF<EventNotification> recalculated = Cf.arrayList();
        for (Tuple2<Option<EventUserWithNotification>, Option<EventUserWithNotification>> c
                : collate(pairs, masterEventId, newEventId))
        {
            DateTimeZone userTz = tzByUid.getOrThrow(c.get1().orElse(c.get2()).get().getEventUser().getUid());
            Option<EventNotification> masterNotification = c.get1().map(EventUserWithNotification::getEventNotification);
            Option<EventNotification> newEventNotification = c.get2().map(EventUserWithNotification::getEventNotification);

            if (!masterNotification.isPresent()) {
                recalculated.add(recalcNextSendTs(newEventNotification.get(), newEvent, userTz, actionInfo));
                continue;
            }
            if (!newEventNotification.isPresent()) {
                recalculated.add(recalcNextSendTs(masterNotification.get(), masterEvent, userTz, actionInfo));
                continue;
            }

            Option<Instant> nextMasterStart = instanceStart(masterEvent, masterNotification.get(), userTz);
            Instant newEventStart = newEvent.getEvent().getStartTs();

            if (!nextMasterStart.isPresent() || !masterEvent.isValidStart(nextMasterStart.get())) {
                recalculated.add(recalcNextSendTs(masterNotification.get(), masterEvent, userTz, actionInfo));
            }
            if (nextMasterStart.isSome(newEventStart)) {
                EventNotification en = newEventNotification.get().copy();
                en.setNextSendTs(masterNotification.get().getNextSendTs());
                recalculated.add(en);
            } else {
                recalculated.add(recalcNextSendTs(newEventNotification.get(), newEvent, userTz, actionInfo));
            }
        }
        notificationDao.updateEventNotificationNextSendTsBatch(recalculated);
    }

    public void recalcAndSaveEventNotifications(
            long eventUserId, ListF<Notification> notifications,
            final EventAndRepetition event, final ActionInfo actionInfo)
    {
        final DateTimeZone userTz = dateTimeManager.getTimeZoneForUid(
                eventUserDao.findEventUserById(eventUserId).getUid());

        notificationDao.saveEventNotificationsBatch(notifications.map(n -> n.toEventNotification(eventUserId))
                .map(en -> recalcNextSendTs(en, event, userTz, actionInfo)));
    }

    private void recalcNextSendTsForEvents(
            ListF<EventAndRepetition> events, ListF<EventUserWithNotifications> pairs, ActionInfo actionInfo)
    {
        MapF<Long, EventAndRepetition> eventById = events.toMapMappingToKey(EventAndRepetition.getEventIdF());

        MapF<PassportUid, DateTimeZone> tzByUid = dateTimeManager.getTimeZonesForUids(
                pairs.map(EventUserWithNotifications.getEventUserF().andThen(EventUser::getUid))).toMap();

        ListF<EventNotification> recalculated = Cf.arrayList();

        for (EventUserWithNotification notification : pairs.flatMap(EventUserWithNotifications.splitF())) {
            EventAndRepetition event = eventById.getOrThrow(notification.getEventUser().getEventId());
            DateTimeZone userTz = tzByUid.getOrThrow(notification.getEventUser().getUid());

            Option<Instant> nextStart = instanceStart(event, notification.getEventNotification(), userTz);

            Option<Instant> sendTs = recalcNextSendTs(
                    notification.getEventNotification(),
                    event.getEvent(), event.getRepetitionInfo(), userTz, actionInfo.getNow());

            if (!nextStart.isPresent()
                || !event.isValidStart(nextStart.get())
                || sendTs.exists(i -> nextStart.get().isAfter(i)))
            {
                recalculated.add(notification.getEventNotification().copy());
                recalculated.last().setNextSendTs(sendTs);
            }
        }
        notificationDao.updateEventNotificationNextSendTsBatch(recalculated);
    }

    static Option<Instant> recalcNextSendTs(
            EventNotification notification,
            Event event, RepetitionInstanceInfo repetitionInfo, DateTimeZone userTz, Instant now)
    {
        Duration offset = Duration.standardMinutes(notification.getOffsetMinute());

        if (event.getIsAllDay()) {
            Option<InstantInterval> nextInstance = RepetitionUtils.getInstanceIntervalStartingAfter(
                    repetitionInfo.changeTz(userTz), now.minus(offset));

            Option<LocalDateTime> nextLocalSend = nextInstance.isPresent()
                    ? Option.of(new LocalDateTime(nextInstance.get().getStart(), userTz).plus(offset))
                    : Option.<LocalDateTime>empty();

            Option<Instant> nextSendTs = nextLocalSend.map(AuxDateTime.toInstantIgnoreGapF2().bind2(userTz));

            if (nextSendTs.exists(ts -> ts.isBefore(now))) {
                return recalcNextSendTs(
                        notification, event, repetitionInfo,
                        userTz, nextInstance.get().getStart().plus(offset).plus(1));
            }
            return nextSendTs;

        } else {
            return RepetitionUtils.getInstanceIntervalStartingAfter(repetitionInfo, now.minus(offset))
                    .map(i -> i.getStart().plus(offset));

        }
    }

    public EventNotification recalcNextSendTs(
            EventNotification en, EventAndRepetition event, DateTimeZone userTz, ActionInfo actionInfo)
    {
        en = en.copy();
        en.setNextSendTs(recalcNextSendTs(en, event.getEvent(), event.getRepetitionInfo(), userTz, actionInfo.getNow()));
        return en;
    }

    public static Option<Instant> instanceStart(EventAndRepetition event, EventNotification notification, DateTimeZone userTz) {
        if (!notification.getNextSendTs().isPresent()) {
            return Option.empty();
        }
        Duration offset = Duration.standardMinutes(notification.getOffsetMinute());

        if (!event.getEvent().getIsAllDay()) {
            return Option.of(notification.getNextSendTs().get().minus(offset));
        }
        LocalDateTime localNextSend = new LocalDateTime(notification.getNextSendTs().get(), userTz);
        return Option.of(AuxDateTime.toInstantIgnoreGap(localNextSend.minus(offset), event.getRepetitionInfo().getTz()));
    }

    private Tuple2List<Option<EventUserWithNotification>, Option<EventUserWithNotification>> collate(
            ListF<EventUserWithNotifications> notifications, long leftEventId, long rightEventId)
    {
        ListF<EventUserWithNotification> split = notifications.flatMap(EventUserWithNotifications.splitF());

        ListF<EventUserWithNotification> leftList = split.filter(en -> en.getEventId() == leftEventId);

        ListF<EventUserWithNotification> rightList = split.filter(en -> en.getEventId() == rightEventId);

        Function<EventUserWithNotification, Tuple3<PassportUid, Channel, Integer>> keyF = eu -> Tuple3.tuple(
                eu.getEventUser().getUid(),
                eu.getEventNotification().getChannel(),
                eu.getEventNotification().getOffsetMinute());

        MapF<Object, ListF<EventUserWithNotification>> leftMap = leftList.groupBy(keyF.uncheckedCast());
        MapF<Object, ListF<EventUserWithNotification>> rightMap = rightList.groupBy(keyF.uncheckedCast());

        Tuple2List<ListF<EventUserWithNotification>, ListF<EventUserWithNotification>> zipped =
                leftMap.keys().plus(rightMap.keys()).toTuple2List(
                        key -> leftMap.getOrElse(key, Cf.list()),
                        key -> rightMap.getOrElse(key, Cf.list()));

        Function2<EventUserWithNotification, EventUserWithNotification,
                Tuple2<EventUserWithNotification, EventUserWithNotification>> tupleF = Tuple2.consF();
        Function<EventUserWithNotification, Option<EventUserWithNotification>> notNullOF = Option::ofNullable;

        Tuple2List<EventUserWithNotification, EventUserWithNotification> result = Tuple2List.arrayList();
        for (Tuple2<ListF<EventUserWithNotification>, ListF<EventUserWithNotification>> t : zipped) {
            result.addAll(t.get1().zip(t.get2()));

            if (t.get1().size() < t.get2().size()) {
                result.addAll(t.get2().subList(t.get1().size(), t.get2().size()).map(tupleF.bind1(null)));
            } else if (t.get2().size() < t.get1().size()) {
                result.addAll(t.get1().subList(t.get2().size(), t.get1().size()).map(tupleF.bind2(null)));
            }
        }
        return result.map1(notNullOF).map2(notNullOF);
    }

    public static ListF<Long> splitOffsets(String offsets) {
        return AuxColl.splitToLongArray(offsets).sorted();
    }

} //~
