package ru.yandex.calendar.logic.notification;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import lombok.Data;
import lombok.With;
import lombok.val;
import org.joda.time.Chronology;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.calendar.frontend.api.mail.IncomingIcsMessageInfo;
import ru.yandex.calendar.frontend.api.mail.XivaReminderSpecificData;
import ru.yandex.calendar.frontend.xiva.Xiva;
import ru.yandex.calendar.frontend.yamb.Yamb;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventNotification;
import ru.yandex.calendar.logic.beans.generated.EventNotificationFields;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.EventUserFields;
import ru.yandex.calendar.logic.beans.generated.NotificationSendStat;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.contact.UnivContact;
import ru.yandex.calendar.logic.domain.DomainManager;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventInterval;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.avail.absence.AbsenceType;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.repetition.EventInstanceInterval;
import ru.yandex.calendar.logic.event.repetition.InfiniteInterval;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.log.EventIdLogDataJson;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.notification.ControlDataNotification.CalendarNotificationType;
import ru.yandex.calendar.logic.notification.xiva.notify.XivaMobileReminderNotifier;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.sending.EventNotificationMessageXmlCreator;
import ru.yandex.calendar.logic.sending.param.MessageParameters;
import ru.yandex.calendar.logic.sending.param.NotificationMessageParameters;
import ru.yandex.calendar.logic.sending.real.MailSender;
import ru.yandex.calendar.logic.sending.real.SmsSender;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.Nick;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.svc.SvcRoutines;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.TlCacheUtils;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.calendar.util.exception.ExceptionUtils;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleExponential;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.commune.json.JsonBoolean;
import ru.yandex.commune.json.JsonNumber;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.inside.passport.PassportSid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.sms.SmsPassportErrorCode;
import ru.yandex.inside.passport.sms.SmsPassportException;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.concurrent.Futures;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.log4j.appender.AppenderContextHolder;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.thread.ThreadLocalTimeoutException;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author Daniel Brylev
 * @author Sergey Shinderuk
 */
public class NotificationSendManager {

    private static final Logger logger = LoggerFactory.getLogger(NotificationSendManager.class);

    @Autowired
    private MailSender mailSender;
    @Autowired
    private SmsSender smsSender;
    @Autowired
    private ControlDataNotification controlDataNotification;
    @Autowired
    private NotificationDao notificationDao;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private RepetitionRoutines repetitionRoutines;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private EventNotificationMessageXmlCreator eventNotificationMessageXmlCreator;
    @Autowired
    private UserManager userManager;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private NotificationSendStatDao notificationSendStatDao;
    @Autowired
    private DomainManager domainManager;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private Xiva xiva;
    @Autowired
    private XivaMobileReminderNotifier xivaMobileReminderNotifier;
    @Autowired
    private Yamb yamb;
    @Autowired
    private SvcRoutines svcRoutines;
    @Autowired
    private EventsLogger eventsLogger;

    private static final Duration EXPIRE_INTERVAL = Duration.standardHours(1);

    private static final Duration ADVANCE_INTERVAL = Duration.standardSeconds(150);

    private static final int NOTIFICATIONS_PER_CALLBACK = 100;

    private static final int SEND_THREADS = 10;

    private static ReschedulePolicy localRetryPolicy = new RescheduleExponential(Duration.standardSeconds(10), 5);


    public ListF<Reminder> getUserDisplayNotifications(PassportUid uid, InfiniteInterval interval, int limit) {
        ListF<Reminder> result = Cf.arrayList();

        ListF<EventNotification> notifications = notificationDao.findEventNotificationsSortedByNextSendTs(
                notificationChannelIs(Channel.DISPLAY).and(notificationNextSendTsIn(interval)),
                eventUserGoesToMeetingAndHasUid(uid));

        for (NotificationInfo info : loadNotificationInfo(notifications)) {
            if (result.size() >= limit) {
                break;
            }
            if (!canNotificationBeShown(info).isPresent()) {
                String eventName = info.getEventWithRelations().getEvent().getName();
                EventNotification notification = info.getNotification();
                Instant notifyTs = notification.getNextSendTs().get();
                Instant eventInstanceStart = notifyTs.minus(Duration.standardMinutes(notification.getOffsetMinute()));
                result.add(new Reminder(eventName, eventInstanceStart, notifyTs));
            }
        }
        return result;
    }

    public void sendNotifications(final ActionInfo actionInfo) {
        NotificationStat stats = new NotificationStat();
        try {
            doSendNotifications(stats, actionInfo);
        } catch (Exception e) {
            ExceptionUtils.rethrowIfTlt(e);
            stats.unexpectedErrors.incrementAndGet();
            throw e;

        } finally {
            logger.info("Notification statistics: emails {} sent, {} failed; sms {} sent, {} failed; " +
                    "total processed {}; unexpectedErrors {}", stats.emailsSent, stats.emailsFailed,
                    stats.smsSent, stats.smsFailed, stats.totalProcessed, stats.unexpectedErrors);

            recordNotificationStat(stats, actionInfo);
        }
    }

    private void doSendNotifications(NotificationStat stats, ActionInfo actionInfo) {
        ListF<CompletableFuture<Void>> undone = Cf.arrayList();

        Function<Function0<CompletableFuture<Void>>, Function0<CompletableFuture<Void>>> withTlsF =
                func -> MasterSlaveContextHolder.withStandardThreadLocalsF(AppenderContextHolder.inheritAppendersF(func));

        Function<Function0V, Function0V> withTlsV = func -> withTlsF.apply(func.asFunction0ReturnNull()).asFunction0V();

        ExecutorService executor = Executors.newFixedThreadPool(SEND_THREADS);
        ScheduledExecutorService scheduler = Executors.newSingleThreadScheduledExecutor();
        try {
            Function2V<Instant, Function0V> schedule = (ts, action) -> scheduler.schedule(
                    withTlsV.apply(() -> executor.submit(withTlsV.apply(TlCacheUtils.withCacheF(action)))),
                    new Duration(Instant.now(), ts).getMillis(), TimeUnit.MILLISECONDS);

            Function<NotificationInfo, Function0<CompletableFuture<Void>>> sendWithRetries =
                    info -> withTlsF.apply(TlCacheUtils.inheritF(() -> sendNotificationWithRetries(info, schedule, stats, actionInfo)));

            Function<ListF<EventNotification>, ListF<Long>> sendAndReturnUndone = ns -> {
                Tuple2List<NotificationInfo, CompletableFuture<Void>> futures = loadNotificationInfo(ns)
                        .zipWith(sendWithRetries)
                        .map2(executor::submit)
                        .map2(Futures::get);

                undone.addAll(futures.get2().filterNot(CompletableFuture::isDone));
                return futures.filterMap(t -> Option.when(!t.get2().isDone(), t.get1().getId()));
            };

            notificationDao.findEventNotificationsSortedByNextSendTs(
                    notificationNextSendTsBefore(actionInfo.getNow().plus(ADVANCE_INTERVAL)),
                    NOTIFICATIONS_PER_CALLBACK, sendAndReturnUndone);

            CompletableFutures.await(CompletableFutures.allOf(undone));

        } finally {
            scheduler.shutdown();
            executor.shutdown();
        }
    }

    private void recordNotificationStat(NotificationStat stat, ActionInfo actionInfo) {
        NotificationSendStat sendStat = new NotificationSendStat();
        sendStat.setTaskReqId(actionInfo.getRequestIdWithHostId());
        sendStat.setTaskStartTs(actionInfo.getNow());
        sendStat.setTaskEndTs(Instant.now());
        sendStat.setEmailsSent(stat.emailsSent.get());
        sendStat.setEmailsFailed(stat.emailsFailed.get());
        sendStat.setSmsSent(stat.smsSent.get());
        sendStat.setSmsFailed(stat.smsFailed.get());
        sendStat.setTotalProcessed(stat.totalProcessed.get());
        sendStat.setUnexpectedErrors(stat.unexpectedErrors.get());

        notificationSendStatDao.saveNotificationSendStat(sendStat);
    }

    private CompletableFuture<Void> sendNotificationWithRetries(
            NotificationInfo notification, Function2V<Instant, Function0V> schedule,
            NotificationStat stats, ActionInfo actionInfo)
    {
        CompletableFuture<Void> future = new CompletableFuture<>();
        try {
            ThreadLocalTimeout.check();
            NotificationSentInfo sent = sendNotification(notification, actionInfo);

            if (sent.result.status == NotificationStatus.TRY_AGAIN) {
                Option<Instant> nextTs = localRetryPolicy.rescheduleAt(Instant.now(), notification.getAttempt() + 1)
                        .filterNot(new Instant(ThreadLocalTimeout.deadlineMillis())::isBefore);

                if (nextTs.isPresent()) {
                    schedule.apply(nextTs.get(), () ->
                            sendNotificationWithRetries(notification.incAttempt(), schedule, stats, actionInfo)
                                    .thenAccept(f -> future.complete(null)));
                } else {
                    future.complete(null);
                }
            } else {
                future.complete(null);
            }

            eventsLogger.log(new NotificationLogEventJson(notification, sent.result), actionInfo);
            stats.add(sent.stat());

            return future;

        } catch (ThreadLocalTimeoutException e) {
            future.completeExceptionally(e);
            throw e;

        } catch (Exception e) {
            logger.error("Error occurred while sending notification", e);
            stats.unexpectedErrors.incrementAndGet();

            future.complete(null);
            return future;

        } catch (Throwable t) {
            future.completeExceptionally(t);
            throw t;
        }
    }

    private NotificationSentInfo sendNotification(NotificationInfo notificationInfo, ActionInfo actionInfo) {
        EventNotification notification = notificationInfo.getNotification();
        Channel channel = notification.getChannel();

        NotificationSendResult status = NotificationSendResult.skipped(canNotificationBeSent(
                notificationInfo, actionInfo.getNow().minus(EXPIRE_INTERVAL)));

        if (!status.skipReason.isPresent()) {
            if (!passportAuthDomainsHolder.containsYandexTeamRu()) {
                notificationInfo = notificationInfo
                        .withSettings(settingsRoutines.getSettingsByUid(notificationInfo.getUid()));
            }
            if (channel == Channel.SVC) {
                status = sendService(notificationInfo, actionInfo);
            } else if (channel == Channel.EMAIL) {
                status = sendEmail(notificationInfo, actionInfo);
            } else if (channel == Channel.SMS) {
                status = sendSms(notificationInfo, actionInfo);
            } else if (channel == Channel.XIVA) {
                status = sendXiva(notificationInfo, actionInfo);
            } else if (channel == Channel.PANEL) {
                status = sendPanel(notificationInfo, actionInfo);
            } else if (channel == Channel.YAMB) {
                status = sendYamb(notificationInfo, actionInfo);
            } else if (channel == Channel.MOBILE) {
                status = sendMobile(notificationInfo, actionInfo);
            }
            logger.info("Sent status: " + status + " " + notification);
        }

        if (status.status != NotificationStatus.TRY_AGAIN) {
            Event event = notificationInfo.getEventWithRelations().getEvent();
            RepetitionInstanceInfo repetitionInfo = notificationInfo.getRepetitionInfo();
            Option<Instant> nextSendTs = NotificationRoutines.recalcNextSendTs(
                    notification, event, repetitionInfo,
                    notificationInfo.getSettings().getTz(),
                    actionInfo.getNow().plus(ADVANCE_INTERVAL));

            notificationDao.updateEventNotificationNextSendTsById(notification.getId(), nextSendTs);
        }
        if (status.status == NotificationStatus.PROCESSED) {
            EventUser eventUser = notificationInfo.getEventUser();
            EventUser eventUserUpdate = new EventUser();

            eventUserUpdate.setId(eventUser.getId());
            eventUserUpdate.setLastSentTs(Instant.now());

            if (status.smsId.isPresent()) {
                eventUserUpdate.setLastSentSmsId(status.smsId);
            }

            eventUserUpdate.setTotalSentCount(eventUser.getTotalSentCount() + 1);

            eventUserDao.updateEventUser(eventUserUpdate, actionInfo);
        }
        return new NotificationSentInfo(notificationInfo, status);
    }

    private NotificationSendResult sendSms(NotificationInfo notificationInfo, ActionInfo actionInfo) {
        Validate.isTrue(notificationInfo.getNotification().getChannel() == Channel.SMS);
        try {
            logger.info("Sending sms to user " + notificationInfo.getSettings().getUid());
            NotificationMessageParameters messageParameters = prepareMessage(notificationInfo);

            String smsId = smsSender.sendNotificationSms(messageParameters, notificationInfo.getSettings().getUid());
            return NotificationSendResult.processed(smsId);

        } catch (SmsPassportException spe) {
            if (spe.getErrorCode() == SmsPassportErrorCode.NOCURRENT) {
                return NotificationSendResult.skipped(Option.of(NotificationSkipReason.NO_CURRENT_PHONE));
            } else {
                ExceptionUtils.rethrowIfTlt(spe);
                logger.error("Error occurred while sending sms", spe);
                return NotificationSendResult.fatalError(spe);
            }
        } catch (Exception e) {
            ExceptionUtils.rethrowIfTlt(e);
            logger.error("Error occurred while sending sms", e);
            return NotificationSendResult.fatalError(e);
        }
    }

    private NotificationSendResult sendEmail(NotificationInfo notificationInfo, ActionInfo actionInfo) {
        Validate.isTrue(notificationInfo.getNotification().getChannel() == Channel.EMAIL);
        try {
            Settings settings = notificationInfo.getSettings().getCommon();
            logger.info("Sending email to user " + settings.getUid() + ", address " + settings.getEmail());
            MessageParameters messageParameters = prepareMessage(notificationInfo)
                    .withGeneratedMessageId(svcRoutines.getSelfUrlHost());

            mailSender.sendEmailsViaTask(Cf.list(messageParameters), actionInfo);
            return NotificationSendResult.processed(
                    settings.getEmail(), messageParameters.getMessageOverrides().messageId.get());

        } catch (Exception e) {
            ExceptionUtils.rethrowIfTlt(e);
            logger.error("Error occurred while sending email", e);
            return NotificationSendResult.fatalError(e);
        }
    }

    private NotificationSendResult sendService(NotificationInfo notificationInfo, ActionInfo actionInfo) {
        Validate.isTrue(notificationInfo.getNotification().getChannel() == Channel.SVC);
        try {
            Settings settings = notificationInfo.getSettings().getCommon();
            Event event = notificationInfo.getEventWithRelations().getEvent();

            PassportSid sid = event.getSid();
            logger.info("Sending " + notificationInfo.getNotification() + " to service " + sid);

            Validate.isTrue(ControlDataNotification.isSupportedBy(sid), "No behavior defined for sid " + sid);
            String externalEventId = notificationInfo.getEventWithRelations().getMainEvent().getExternalId();

            Tuple2<NotificationStatus, Option<String>> status = controlDataNotification.send(
                    event, settings.getUid(), externalEventId, actionInfo.getNow(), CalendarNotificationType.NOTIFY);
            if (status.get1() != NotificationStatus.PROCESSED) {
                logger.warn("Could not call to service sid " + sid + ". Reason: " + status.get2().getOrNull());
            }
            return NotificationSendResult.cons(status.get1());

        } catch (Exception e) {
            ExceptionUtils.rethrowIfTlt(e);
            logger.error("Error occurred while notifying service", e);
            return NotificationSendResult.fatalError(e);
        }
    }

    private NotificationSendResult sendXiva(NotificationInfo notification, ActionInfo actionInfo) {
        Validate.equals(Channel.XIVA, notification.getNotification().getChannel());

        return sendOrFatal(notification, () ->
                xiva.push(notification.getUid(), "meeting-reminder", prepareMailMeetingReminderMessage(notification)));
    }

    private NotificationSendResult sendMobile(NotificationInfo notification, ActionInfo actionInfo) {
        Validate.equals(Channel.MOBILE, notification.getNotification().getChannel());

        return sendOrFatal(notification, () ->
                xivaMobileReminderNotifier.notifyOrThrow(
                        notification.userInfo.getUid(),
                        notification.eventWithRelations
                ));
    }

    private NotificationSendResult sendPanel(NotificationInfo notification, ActionInfo actionInfo) {
        Validate.equals(Channel.PANEL, notification.getNotification().getChannel());

        return sendOrFatal(notification, () ->
                xiva.push(notification.getUid(), "panel-reminder", preparePanelEventReminderMessage(notification)));
    }

    private NotificationSendResult sendYamb(NotificationInfo notification, ActionInfo actionInfo) {
        Validate.equals(Channel.YAMB, notification.getNotification().getChannel());
        try {
            logger.info("Sending {} notification to user {}", Channel.YAMB, notification.getUid());
            yamb.sendMessage(notification.getUid(), prepareYambNotificationMessage(notification));

            return NotificationSendResult.cons(NotificationStatus.PROCESSED);

        } catch (Exception e) {
            ExceptionUtils.rethrowIfTlt(e);
            logger.error("Error occurred while sending {} notification", Channel.YAMB, e);
            return NotificationSendResult.tryAgain(e);
        }
    }

    private NotificationSendResult sendOrFatal(NotificationInfo notification, Function0V sender) {
        String channel = notification.getNotification().getChannel().name().toLowerCase();
        try {
            logger.info("Sending {} notification to user {}", channel, notification.getUid());
            sender.apply();
            return NotificationSendResult.cons(NotificationStatus.PROCESSED);

        } catch (Exception e) {
            ExceptionUtils.rethrowIfTlt(e);
            logger.error("Error occurred while sending {} notification", channel, e);
            return NotificationSendResult.fatalError(e);
        }
    }

    private NotificationMessageParameters prepareMessage(NotificationInfo notificationInfo) {
        Settings settings = notificationInfo.getSettings().getCommon();

        PassportUid uid = settings.getUid();
        String nick = Nick.getRecipient(settings);
        UnivContact recipient = new UnivContact(settings.getEmail(), nick, Option.of(uid));

        DateTimeZone timezone = dateTimeManager.getTimeZoneForUid(settings.getUid());

        EventNotification ntf = notificationInfo.getNotification();
        Instant eventInstanceStart = notificationInfo.getInstanceStartTs();

        // use correct today for notifications sending in advance
        Instant now = ObjectUtils.max(Instant.now(), ntf.getNextSendTs().get());

        return eventNotificationMessageXmlCreator.createNotificationMessageParameters(
                uid, recipient, notificationInfo.getEventWithRelations(),
                notificationInfo.getRepetitionInfo(), eventInstanceStart, now, timezone);
    }

    private String prepareMailMeetingReminderMessage(NotificationInfo notificationInfo) { // CAL-6223
        Settings settings = notificationInfo.getSettings().getCommon();

        EventWithRelations event = notificationInfo.getEventWithRelations();

        ListF<ResourceInfo> resources = event.getResources().isNotEmpty()
                ? resourceRoutines.selectResourcesFromUserOfficeOrCityOrGetAll(settings.getUid(), event.getResources())
                : Cf.list();

        return new IncomingIcsMessageInfo(
                Optional.of(new XivaReminderSpecificData(
                        notificationInfo.getEvent().getId(),
                        notificationInfo.eventWithRelations.getExternalId(),
                        notificationInfo.getInstanceStartTs())),
                event.getEvent().getName(), event.getEvent().getLocation(), resources,
                notificationInfo.getInstanceInterval(), Optional.empty(), settings.getLanguage()).serializeToJson();
    }

    private String preparePanelEventReminderMessage(NotificationInfo notificationInfo) { // PP-61
        Settings settings = notificationInfo.getSettings().getCommon();
        DateTimeZone userTz = AuxDateTime.getVerifyDateTimeZone(settings.getTimezoneJavaid());
        EventWithRelations event = notificationInfo.getEventWithRelations();

        Tuple2List<String, JsonValue> fields = Tuple2List.arrayList();
        fields.add("name", JsonString.valueOf(event.getEvent().getName()));
        fields.add("description", JsonString.valueOf(event.getEvent().getDescription()));
        fields.add("location", JsonString.valueOf(event.getEvent().getLocation()));

        DateTimeFormatter format = ISODateTimeFormat.dateTimeNoMillis();
        EventInterval interval = event.getEventInterval(notificationInfo.getInstanceInterval());

        fields.add("startTs", JsonNumber.valueOf(interval.getStart().toInstant(userTz).getMillis()));
        fields.add("endTs", JsonNumber.valueOf(interval.getEnd().toInstant(userTz).getMillis()));
        fields.add("isAllDay", JsonBoolean.valueOf(event.getEvent().getIsAllDay()));

        fields.add("startTime", JsonString.valueOf(interval.getStart().toLocalDateTime(userTz).toString(format)));
        fields.add("endTime", JsonString.valueOf(interval.getEnd().toLocalDateTime(userTz).toString(format)));

        LocalDateTime eventDate = new LocalDateTime(notificationInfo.getInstanceStartTs(), userTz);
        String url = UriBuilder.cons(svcRoutines.getCalendarUrlBySettings(settings)).appendPath("event")
                .addParam("event_id", event.getId())
                .addParam("event_date", eventDate.toString(format)).toUrl();

        fields.add("url", JsonString.valueOf(url));

        return new JsonObject(fields).serialize();
    }

    private String prepareYambNotificationMessage(NotificationInfo notification) {
        return prepareMessage(notification).formatYamb();
    }

    private ListF<NotificationInfo> loadNotificationInfo(ListF<EventNotification> notifications) {
        ListF<Long> eventUserIds = notifications.map(EventNotification.getEventUserIdF()).stableUnique();
        ListF<EventUser> eventUsers = eventUserDao.findEventUsersByIds(eventUserIds);
        ListF<PassportUid> uids = eventUsers.map(EventUser.getUidF());
        ListF<Long> eventIds = eventUsers.map(EventUser.getEventIdF());

        MapF<Long, EventUser> eventUserById = eventUsers.toMapMappingToKey(EventUser.getIdF());

        MapF<PassportUid, UserInfo> userInfoByUid = userManager.getUserInfos(uids).toMapMappingToKey(UserInfo.getUidF());

        MapF<PassportUid, SettingsInfo> settingsByUid = !passportAuthDomainsHolder.containsYandexTeamRu()
                ? settingsRoutines.getSettingsByUidIfExistsNoCacheNoUpdate(uids)
                : settingsRoutines.getSettingsByUidIfExistsBatch(uids);

        MapF<Long, EventWithRelations> eventById = eventDbManager.getEventsWithRelationsByIdsSafe(eventIds)
                .toMapMappingToKey(EventWithRelations::getId);

        // GREG-937: filter reperition that are far than half an hour from now
        MapF<Long, RepetitionInstanceInfo> repetitionInfoByEventId = repetitionRoutines.getRepetitionInstanceInfos(
                eventById.values().toList())
                .mapValues(repetitionInstanceInfo -> repetitionInstanceInfo.withoutPastRecurrencesAndExdates(Instant.now().minus(Duration.standardMinutes(30))));

        ListF<NotificationInfo> result = Cf.arrayList();
        for (EventNotification notification : notifications) {
            if (!eventUserById.containsKeyTs(notification.getEventUserId())) {
                logger.warn("Cannot load EventUser. Skip " + notification);
                continue;
            }
            EventUser eventUser = eventUserById.getTs(notification.getEventUserId());

            if (!settingsByUid.containsKeyTs(eventUser.getUid())) {
                logger.warn("Cannot load Settings. Skip " + notification);
                continue;
            }
            SettingsInfo settings = settingsByUid.getTs(eventUser.getUid());

            if (!eventById.containsKeyTs(eventUser.getEventId())) {
                logger.warn("Cannot load EventWithRelations. Skip " + notification);
                continue;
            }
            EventWithRelations event = eventById.getTs(eventUser.getEventId());

            if (!repetitionInfoByEventId.containsKeyTs(eventUser.getEventId())) {
                logger.warn("Cannot load RepetitionInstanceInfo. Skip " + notification);
                continue;
            }
            RepetitionInstanceInfo repetitionInfo = repetitionInfoByEventId.getTs(eventUser.getEventId());

            UserInfo userInfo = userInfoByUid.getTs(eventUser.getUid());
            result.add(new NotificationInfo(notification, eventUser, userInfo, settings, event, repetitionInfo, 0));
        }
        return result;
    }

    private Option<NotificationSkipReason> canNotificationBeSent(NotificationInfo info, Instant expiredSince) {
        EventNotification notification = info.getNotification();

        long eventId = info.getEventWithRelations().getEvent().getId();
        PassportUid uid = info.getSettings().getUid();
        logger.info("Checking notification to be sent. Event id: " + eventId + " User: " + uid + " " + notification);

        Option<NotificationSkipReason> reason = canNotificationBeSentOrShown(info);
        if (reason.isPresent()) return reason;

        if (notification.getChannel() == Channel.SMS && !domainManager.hasMobilePhone(uid)) {
            logger.info("Notification cannot be sent because user has no current phone");
            return Option.of(NotificationSkipReason.NO_CURRENT_PHONE);
        }

        if (notification.getNextSendTs().get().isBefore(expiredSince)) {
            logger.info("Notification cannot be sent because it is expired");
            return Option.of(NotificationSkipReason.EXPIRED);
        }

        if (notification.getChannel() == Channel.XIVA) { // CAL-6223
            if (info.getEventWithRelations().getParticipants().getUserParticipantsSafeWithInconsistent().size() < 2) {
                logger.info("Notification cannot be sent because it is not about meeting");
                return Option.of(NotificationSkipReason.NOT_MEETING);
            }
            if (info.getSettings().getYt().exists(SettingsYt.getXivaReminderEnabledF().andThenEquals(false))) {
                logger.info("Notification cannot be sent because user disabled reminders via xiva");
                return Option.of(NotificationSkipReason.XIVA_DISABLED);
            }
        }
        if (notification.getChannel() == Channel.PANEL) {
            if (!Cf.list(EventType.USER, EventType.FEED).containsTs(info.getEvent().getType())) {
                logger.info("Notification cannot be sent because event type is {}", info.getEvent().getType());
                return Option.of(NotificationSkipReason.TYPE_MISMATCH);
            }
        }
        if (passportAuthDomainsHolder.containsPublic() && !userManager.getUserByUid(info.getUid()).isPresent()) {
            logger.info("Notification cannot be sent because user is deleted"); // CAL-7055
            return Option.of(NotificationSkipReason.DELETED_USER);
        }

        return Option.empty();
    }

    private Option<NotificationSkipReason> canNotificationBeShown(NotificationInfo info) {
        logger.info("Checking notification to be shown " + info.getNotification());
        return canNotificationBeSentOrShown(info);
    }

    private Option<NotificationSkipReason> canNotificationBeSentOrShown(NotificationInfo info) {
        EventNotification notification = info.getNotification();
        EventWithRelations event = info.getEventWithRelations();
        EventUser eventUser = info.getEventUser();

        SettingsInfo settingsInfo = info.getSettings();
        Settings settings = settingsInfo.getCommon();

        if (!notification.getNextSendTs().isPresent()) {
            logger.info("Notification cannot be sent or shown because of unknown send time");
            return Option.of(NotificationSkipReason.UNKNOWN_SEND_TIME);
        }
        boolean remindUndecided = notification.getChannel() == Channel.XIVA
                || notification.getChannel() == Channel.YAMB
                || settingsInfo.getYt().exists(SettingsYt::getRemindUndecided);

        ListF<Decision> refusingDecisions = !remindUndecided
                ? Cf.list(Decision.UNDECIDED, Decision.NO)
                : Cf.list(Decision.NO);

        if (refusingDecisions.containsTs(eventUser.getDecision())) {
            logger.info("Notification cannot be sent or shown because user does not go");
            return Option.of(NotificationSkipReason.NOT_ACCEPTED);
        }
        final Chronology userTimeZoneChronology = AuxDateTime.getVerifyChrono(settings.getTimezoneJavaid());
        if (Cf.set(Channel.EMAIL, Channel.SMS, Channel.DISPLAY, Channel.YAMB).containsTs(notification.getChannel())) {
            // Global-off check
            if (settings.getNoNtfStartTs().isPresent() || settings.getNoNtfEndTs().isPresent()) {
                Instant sendTs = notification.getNextSendTs().get();
                Option<Instant> noNtfStartO = settings.getNoNtfStartTs();
                Option<Instant> noNtfEndO = settings.getNoNtfEndTs();
                if (noNtfEndO.isPresent()) {
                    // CAL-1461: add 1 day to the database value for 'end date'.
                    // Also look at CmdCleanTables (where we clean no-ntf intervals).
                    noNtfEndO = Option.of(AuxDateTime.getNextDayMs(noNtfEndO.get(), userTimeZoneChronology.getZone()));
                }
                if ((!noNtfStartO.isPresent() || !sendTs.isBefore(noNtfStartO.get())) &&
                    (!noNtfEndO.isPresent() || sendTs.isBefore(noNtfEndO.get()))) {
                    logger.info("Notification cannot be sent or shown because user turned off notifications" +
                            (noNtfStartO.isPresent() ? " since " + noNtfStartO.get() : "") +
                            (noNtfEndO.isPresent() ? " until " + noNtfEndO.get() : ""));
                    return Option.of(NotificationSkipReason.TUNED_OFF);
                }
            }
        }
        if (Cf.set(Channel.SMS, Channel.DISPLAY).containsTs(notification.getChannel())) {
            // Local-time-off check
            if (settings.getNoNtfStartTm().isPresent() || settings.getNoNtfEndTm().isPresent()) {
                LocalTime localSendTime = new LocalTime(notification.getNextSendTs().get(), userTimeZoneChronology);
                int sendMs = localSendTime.getMillisOfDay();
                int startMs = settings.getNoNtfStartTm().getOrElse(0);
                int endMs = settings.getNoNtfEndTm().getOrElse(0);
                if ( // Condition is somewhat redundant but more easy for understanding (its quick)
                    (startMs < endMs && (startMs <= sendMs && sendMs < endMs)) || // normal case
                    (endMs < startMs && (startMs <= sendMs || sendMs < endMs)) // midnight case
                    // startMs == endMs - no case, because no restriction
                ) {
                    logger.info("Notification cannot be sent or shown because user turned off notifications" +
                            " since " + AuxDateTime.formatOffset(startMs) +
                            " until " + AuxDateTime.formatOffset(endMs));
                    return Option.of(NotificationSkipReason.TUNED_OFF);
                }
            }
        }
        if (Cf.set(Channel.SMS, Channel.DISPLAY, Channel.XIVA, Channel.YAMB).containsTs(notification.getChannel())
            && settingsInfo.getYt().exists(Function1B.wrap(SettingsYt.getNoNtfDuringAbsenceF())))
        {
            ListF<Event> absences = eventRoutines.findNowEventsOnAbsenceLayer(
                    settings.getUid(), notification.getNextSendTs().get()).map(EventInstanceInterval.getEventF());

            if (absences.exists(Event.getNameF().andThen(AbsenceType.byEventNameF()).andThen(
                    Cf2.isSomeOfF(Cf.list(AbsenceType.VACATION, AbsenceType.ILLNESS)))))
            {
                logger.info("Notification cannot be sent or shown because user is absent"); // CAL-6683
                return Option.of(NotificationSkipReason.ABSENT_USER);
            }
        }
        if (settingsInfo.getYt().exists(Function1B.wrap(SettingsYt.getIsDismissedF()))) {
            // https://jira.yandex-team.ru/browse/CAL-4330
            logger.info("Notification cannot be sent or shown because user was dismissed");
            return Option.of(NotificationSkipReason.DISMISSED_USER);
        }
        if (!eventRoutines.findNotDeclinedUserLayerWithEvent(
                info.getUserInfo().getUid(), info.getEventWithRelations()).isPresent())
        {
            logger.info("Notification cannot be sent or shown because user does not see event on any of his layers");
            return Option.of(NotificationSkipReason.NO_LAYER);
        }
        val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(info.getUserInfo(), event);
        if (!authorizer.canViewEvent(info.getUserInfo(), eventAuthInfo, ActionSource.notTrusted())) {
            logger.info("Notification cannot be sent or shown because user has no permissions to view event");
            return Option.of(NotificationSkipReason.NO_PERMISSIONS);
        }
        return Option.empty();
    }

    private static SqlCondition eventUserGoesToMeeting() {
        return EventUserFields.DECISION.eq(Decision.YES).or(EventUserFields.DECISION.eq(Decision.MAYBE));
    }

    private static SqlCondition eventUserGoesToMeetingAndHasUid(PassportUid uid) {
        return eventUserGoesToMeeting().and(EventUserFields.UID.eq(uid));
    }

    private static SqlCondition notificationNextSendTsIn(InfiniteInterval interval) {
        Instant start = interval.getStart();
        Option<Instant> endO = interval.getEnd();
        return !endO.isPresent() ?
                EventNotificationFields.NEXT_SEND_TS.ge(start) :
                EventNotificationFields.NEXT_SEND_TS.ge(start).and(EventNotificationFields.NEXT_SEND_TS.lt(endO.get()));
    }

    private static SqlCondition notificationChannelIs(Channel channel) {
        return EventNotificationFields.CHANNEL.eq(channel);
    }

    private static SqlCondition notificationNextSendTsBefore(Instant maxNextSendTs) {
        return EventNotificationFields.NEXT_SEND_TS.lt(maxNextSendTs);
    }

    @Data
    @With
    public static class NotificationInfo {
        private final EventNotification notification;
        private final EventUser eventUser;
        private final UserInfo userInfo;
        private final SettingsInfo settings;
        private final EventWithRelations eventWithRelations;
        private final RepetitionInstanceInfo repetitionInfo;
        private final int attempt;

        public long getId() {
            return notification.getId();
        }

        public PassportUid getUid() {
            return settings.getUid();
        }

        public Instant getInstanceStartTs() {
            EventAndRepetition event = new EventAndRepetition(eventWithRelations.getEvent(), repetitionInfo);
            return NotificationRoutines.instanceStart(event, notification, settings.getTz()).get();
        }

        public InstantInterval getInstanceInterval() {
            return RepetitionUtils.getClosestInstanceInterval(repetitionInfo, getInstanceStartTs())
                    .getOrElse(new InstantInterval(getInstanceStartTs(), repetitionInfo.getEventInterval().getDuration()));
        }

        public Event getEvent() {
            return eventWithRelations.getEvent();
        }

        public EventIdLogDataJson getEventIdLogData() {
            return new EventIdLogDataJson(eventWithRelations);
        }

        public NotificationInfo incAttempt() {
            return new NotificationInfo(
                    notification, eventUser, userInfo, settings,
                    eventWithRelations, repetitionInfo, attempt + 1);
        }
    }

    @Data
    public static class NotificationSentInfo {
        private final NotificationInfo notification;
        private final NotificationSendResult result;

        public NotificationSentInfo(NotificationInfo notification, NotificationSendResult result) {
            this.notification = notification;
            this.result = result;
        }

        public NotificationStat stat() {
            NotificationStat stats = new NotificationStat();

            Channel channel = notification.getNotification().getChannel();
            NotificationStatus status = result.status;

            if (channel == Channel.EMAIL) {
                if (status == NotificationStatus.PROCESSED) stats.emailsSent.incrementAndGet();
                else if (status == NotificationStatus.FATAL_ERROR) stats.emailsFailed.incrementAndGet();

            } else if (channel == Channel.SMS) {
                if (status == NotificationStatus.PROCESSED) stats.smsSent.incrementAndGet();
                else if (status == NotificationStatus.FATAL_ERROR) stats.smsFailed.incrementAndGet();
            }
            if (status != NotificationStatus.TRY_AGAIN) {
                stats.totalProcessed.incrementAndGet();
            }
            return stats;
        }
    }

    private static class NotificationStat {
        public final AtomicInteger emailsSent = new AtomicInteger();
        public final AtomicInteger emailsFailed = new AtomicInteger();
        public final AtomicInteger smsSent = new AtomicInteger();
        public final AtomicInteger smsFailed = new AtomicInteger();
        public final AtomicInteger totalProcessed = new AtomicInteger();
        public final AtomicInteger unexpectedErrors = new AtomicInteger();

        void add(NotificationStat other) {
            emailsSent.addAndGet(other.emailsSent.get());
            emailsFailed.addAndGet(other.emailsFailed.get());
            smsSent.addAndGet(other.smsSent.get());
            smsFailed.addAndGet(other.smsFailed.get());
            totalProcessed.addAndGet(other.totalProcessed.get());
            unexpectedErrors.addAndGet(other.unexpectedErrors.get());
        }
    }

    void setYambForTest(Yamb yamb) {
        this.yamb = yamb;
    }

    static ReschedulePolicy getLocalRetryPolicy() {
        return localRetryPolicy;
    }

    static void setLocalRetryPolicyForTest(ReschedulePolicy policy) {
        localRetryPolicy = policy;
    }
}
