package ru.yandex.calendar.logic.notification;

import lombok.AllArgsConstructor;
import lombok.Getter;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.misc.lang.Validate;

public abstract class NotificationsData {
    public abstract static class Create extends NotificationsData {
    }

    public static class UseLayerDefaultIfCreate extends Create {
    }

    @Getter
    @AllArgsConstructor
    public static class CreateWithData extends Create {
        private final ListF<Notification> notifications;
        private final Option<SetF<Channel>> knownChannels;
    }

    @AllArgsConstructor
    public static class Update extends NotificationsData {
        private final MapF<Channel, ListF<Duration>> channelOffsets;

        public SetF<Channel> getChannelsToUpdate() {
            return channelOffsets.keySet();
        }

        public ListF<Notification> getNotifications() {
            ListF<Notification> notifications = Cf.arrayList();
            for (Channel channel : channelOffsets.keySet()) {
                for (Duration duration : channelOffsets.getTs(channel)) {
                    notifications.add(new Notification(channel, duration));
                }
            }
            return notifications;
        }
    }

    public static final SetF<Channel> WEB_CHANNELS = Cf.set(
            Channel.EMAIL, Channel.SMS, Channel.SVC, Channel.DISPLAY, Channel.YAMB);

    public static Update updateFromWeb(ListF<Notification> notifications) {
        return updateChannels(WEB_CHANNELS, notifications);
    }

    private static final SetF<Channel> icsChannels = Cf.set(Channel.DISPLAY, Channel.AUDIO);
    public static Update updateFromIcs(ListF<Notification> notifications) {
        return updateChannels(icsChannels, notifications);
    }

    public static final SetF<Channel> mailChannels = Cf.set(Channel.SMS, Channel.EMAIL);
    public static Update updateFromMail(ListF<Notification> notifications) {
        return updateChannels(mailChannels, notifications);
    }

    public static Update updateChannels(SetF<Channel> allChannels, ListF<Notification> notifications) {
        Validate.isEmpty(notifications.map(Notification::getChannel).unique().minus(allChannels));

        MapF<Channel, ListF<Duration>> channelOffsets = Cf.toHashMap(empty(allChannels.toList()));
        MapF<Channel, ListF<Notification>> x = notifications.groupBy(Notification::getChannel);
        for (Channel channel : x.keySet()) {
            channelOffsets.put(channel, x.getTs(channel).map(Notification::getOffset));
        }
        return new Update(channelOffsets);
    }

    public static Create createFromIcs(ListF<Notification> notifications) {
        if (notifications.isEmpty()) {
            return useLayerDefaultIfCreate();
        } else {
            return new CreateWithData(notifications, Option.of(icsChannels));
        }
    }

    public static Create createFromMail(ListF<Notification> notifications) {
        return new CreateWithData(notifications, Option.of(mailChannels));
    }

    public static Create createFromWeb(ListF<Notification> notifications) {
        return new CreateWithData(notifications, Option.of(WEB_CHANNELS));
    }

    public static CreateWithData create(ListF<Notification> notifications) {
        return new CreateWithData(notifications, Option.empty());
    }

    public static CreateWithData createEmpty() {
        return create(Cf.list());
    }

    public static Update turnedOff() {
        ListF<Channel> channels = Channel.R.valuesList();
        return new Update(empty(channels));
    }

    public static Update notChanged() {
        return new Update(Cf.map());
    }

    private static MapF<Channel, ListF<Duration>> empty(ListF<Channel> channels) {
        return channels.toMapMappingToValue(Function.constF(Cf.list()));
    }

    public boolean isCreate() {
        return this instanceof Create;
    }

    public boolean isCreateWithData() {
        return this instanceof CreateWithData;
    }

    public static UseLayerDefaultIfCreate useLayerDefaultIfCreate() {
        return new UseLayerDefaultIfCreate();
    }

    public CreateWithData asCreateWithData() {
        return (CreateWithData) this;
    }

    public boolean isUseLayerDefaultIfCreate() {
        return this instanceof UseLayerDefaultIfCreate;
    }
}
