package ru.yandex.calendar.logic.notification;

import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;
import org.joda.time.format.DateTimeFormat;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.beans.generated.SendingSms;
import ru.yandex.calendar.logic.beans.generated.SendingSmsFields;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.sending.real.SmsSender;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.exception.ExceptionUtils;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.sms.SmsPassportErrorCode;
import ru.yandex.inside.passport.sms.SmsPassportException;
import ru.yandex.misc.TranslitUtils;
import ru.yandex.misc.db.masterSlave.MasterSlaveUnitUnavailableException;
import ru.yandex.misc.digest.Sha256;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author dbrylev
 */
public class SmsNotificationManager {

    private static final Logger logger = LoggerFactory.getLogger(SmsNotificationManager.class);

    @Autowired
    private SendingSmsDao sendingSmsDao;
    @Autowired
    private UserManager userManager;
    @Autowired
    private SmsSender smsSender;


    public void submitEventDeletionSmss(
            PassportUid actorUid, final Event event, final MainEvent mainEvent, final Instant instanceStart,
            ListF<SettingsInfo> recipients, final ActionInfo actionInfo)
    {
        final Instant now = actionInfo.getNow();

        final Option<String> actorLogin = userManager.getYtUserLoginByUid(actorUid);
        final Function<SettingsInfo, String> messageF = new Function<SettingsInfo, String>() {
            public String apply(SettingsInfo settings) {
                String message = "Встреча \"" + event.getName() + "\"" +
                        " в " + new LocalTime(instanceStart, settings.getTz()).toString("HH:mm") +
                        " отменена" + (actorLogin.isPresent() ? " " + actorLogin.get() + "@" : "");

                return settings.getCommon().getTranslitSms() ? TranslitUtils.translit(message) : message;
            }
        };

        final Instant deadline = ObjectUtils.max(actionInfo.getNow().plus(Duration.standardMinutes(15)), instanceStart);

        final String activeId = Cf.list("delete",
                new LocalDate(instanceStart, DateTimeZone.forID(mainEvent.getTimezoneId())),
                Sha256.A.digest(mainEvent.getExternalId()).base64()).mkString("_");

        sendingSmsDao.saveIgnoreDuplicates(recipients.map(new Function<SettingsInfo, SendingSms>() {
            public SendingSms apply(SettingsInfo s) {
                SendingSms sms = new SendingSms();
                sms.setUid(s.getUid());
                sms.setActiveId(StringUtils.take(activeId, SendingSmsFields.ACTIVE_ID.getCut().getOrElse(100500)));

                sms.setMessage(messageF.apply(s));

                sms.setSubmitTs(now);
                sms.setSubmitReqId(actionInfo.getRequestIdWithHostId());
                sms.setExpirationTs(deadline);

                return sms;
            }
        }));
    }

    public void submitEventMovingSmss(
            PassportUid actorUid, Event event, MainEvent mainEvent,
            Instant newStart, ListF<SettingsInfo> recipients, ActionInfo actionInfo)
    {
        Instant now = actionInfo.getNow();

        Option<String> actorLogin = userManager.getYtUserLoginByUid(actorUid);
        Function<SettingsInfo, String> messageF = settings -> {
            String message = "Встреча \"" + event.getName() + "\"" +
                    " перенесена на " + new LocalDateTime(newStart, settings.getTz()).toString("HH:mm dd/MM") +
                    (actorLogin.isPresent() ? " " + actorLogin.get() + "@" : "");

            return settings.getCommon().getTranslitSms() ? TranslitUtils.translit(message) : message;
        };

        Instant deadline = ObjectUtils.max(actionInfo.getNow().plus(Duration.standardMinutes(15)), newStart);

        String activeId = Cf.list("move",
                newStart.toString(DateTimeFormat.forPattern("dd/MM HH:mm")),
                event.getRecurrenceId().isPresent()
                        ? new LocalDate(event.getRecurrenceId().get(), DateTimeZone.UTC).toString()
                        : "",
                Sha256.A.digest(mainEvent.getExternalId()).base64()).mkString("_");

        sendingSmsDao.saveIgnoreDuplicates(recipients.map(new Function<SettingsInfo, SendingSms>() {
            public SendingSms apply(SettingsInfo s) {
                SendingSms sms = new SendingSms();
                sms.setUid(s.getUid());
                sms.setActiveId(StringUtils.take(activeId, SendingSmsFields.ACTIVE_ID.getCut().getOrElse(100500)));

                sms.setMessage(messageF.apply(s));

                sms.setSubmitTs(now);
                sms.setSubmitReqId(actionInfo.getRequestIdWithHostId());
                sms.setExpirationTs(deadline);

                return sms;
            }
        }));
    }

    public void sendSubmittedSmss(final ActionInfo actionInfo) {
        sendingSmsDao.findNotProcessed(new Function1V<SendingSms>() {
            public void apply(SendingSms sms) {
                sendSms(sms, actionInfo);
            }
        });
    }

    private void sendSms(SendingSms sms, ActionInfo actionInfo) {
        logger.info("Going to send sms {} to user {}", sms.getActiveId(), sms.getUid());

        SendingSms update = new SendingSms();

        if (sms.getExpirationTs().isAfter(actionInfo.getNow())) {
            Tuple2<NotificationStatus, String> result = doSendSms(sms);

            if (NotificationStatus.PROCESSED == result.get1()) {
                update.setSentSmsId(result.get2());
                update.setSentTs(actionInfo.getNow());
                update.setIsProcessed(true);

            } else {
                update.setSentFailReason(StringUtils.take(
                        result.get2(), SendingSmsFields.SENT_FAIL_REASON.getCut().getOrElse(100500)));
                update.setIsProcessed(NotificationStatus.TRY_AGAIN != result.get1());
            }
        } else {
            update.setSentFailReason("expired");
            update.setIsProcessed(true);
        }

        try {
            sendingSmsDao.updateByUidAndActiveId(sms.getUid(), sms.getActiveId(), update);

        } catch (MasterSlaveUnitUnavailableException ex) {
            throw ex;
        } catch (Exception ex) {
            ExceptionUtils.rethrowIfTlt(ex);
            logger.error("Error occurred while updating sms status " + ex, ex);
        }
    }

    private Tuple2<NotificationStatus, String> doSendSms(SendingSms sms) {
        try {
            return Tuple2.tuple(NotificationStatus.PROCESSED, smsSender.send(sms.getMessage(), sms.getUid()));

        } catch (SmsPassportException ex) {
            logger.error("Passport error occurred while sending sms " + ex);

            if (Cf.list(
                    SmsPassportErrorCode.PERMANENTBLOCK, SmsPassportErrorCode.PHONEBLOCKED,
                    SmsPassportErrorCode.NOCURRENT, SmsPassportErrorCode.NOROUTE).containsTs(ex.getErrorCode()))
            {
                return Tuple2.tuple(NotificationStatus.FATAL_ERROR, SmsPassportErrorCode.R.toXmlName(ex.getErrorCode()));
            } else {
                return Tuple2.tuple(NotificationStatus.TRY_AGAIN, SmsPassportErrorCode.R.toXmlName(ex.getErrorCode()));
            }
        } catch (Exception ex) {
            logger.error("Error occurred while sending sms " + ex, ex);
            return Tuple2.tuple(NotificationStatus.TRY_AGAIN, ex.getClass().getSimpleName());
        }
    }
}
