package ru.yandex.calendar.logic.notification.xiva;

import java.util.Optional;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.event.ModificationInfo;
import ru.yandex.calendar.logic.event.RecurrenceIdOrMainEvent;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.repetition.EventInstanceInterval;
import ru.yandex.calendar.logic.notification.xiva.content.XivaNotificationType;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.inside.passport.PassportUid;

public class NotificationPreparedDataBuilder {
    @Autowired
    private EventUserDao eventUsers;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private UserManager userManager;

    public NotificationPreparedData build(EventAndRepetition event, Optional<ModificationInfo> modificationInfo, XivaNotificationType type) {
        Optional<Instant> oldStartTs = calculateEventTimeBeforeUpdate(event, modificationInfo, type);
        EventAndRepetition currentEvent = calculateCurrentEvent(event, modificationInfo, type);
        EventInstanceInterval closestEventInterval = currentEvent.getClosestIntervalWithRecurrence(Instant.now()).get();
        Event closestEvent = calculateClosestEvent(currentEvent, closestEventInterval);
        Optional<NameI18n> organizer = loadOrganizerName(currentEvent); //organizer may be changed
        int ttl = calculateTtl(closestEventInterval.getStart(), type);

        return new NotificationPreparedData(
                organizer,
                closestEventInterval.getStart(),
                oldStartTs,
                closestEvent.getId(),
                closestEvent.getName(),
                closestEvent.getType(),
                type,
                ttl);
    }

    /**
     * In case of single event - it's just a new state of event
     * In case of recurring event - more interesting
     *
     * @param event               - created event or old state of updated\removed event
     * @param modificationInfoOpt - information about event modification
     * @param type                - action type
     * @return
     */
    private EventAndRepetition calculateCurrentEvent(EventAndRepetition event, Optional<ModificationInfo> modificationInfoOpt, XivaNotificationType type) {
        if (type == XivaNotificationType.UPDATED) {
            if (modificationInfoOpt.isEmpty()) {
                return event;
            }

            ModificationInfo modificationInfo = modificationInfoOpt.get();
            switch (modificationInfo.getScope()) {
                case ALL:
                    return modificationInfo.getUpdatedEvent().get();
                case SINGLE:
                    EventAndRepetition updatedEvent = modificationInfo.getUpdatedEvent().get();
                    return modificationInfo.getNewEvent()
                            .orElse(updatedEvent);
                default:
                    Option<EventAndRepetition> possibleCurrentEvent = modificationInfo.getNewEvent()
                            .orElse(modificationInfo.getUpdatedEvent());

                    if (possibleCurrentEvent.isPresent()) {
                        return possibleCurrentEvent.get();
                    }
            }
        }
        return event;
    }

    private Event calculateClosestEvent(EventAndRepetition event, EventInstanceInterval closestInterval) {
        return event.getEvent();
    }

    /**
     * In case of single event - it's just old state of event (for update\delete actions)
     * In case of recurring event - more interesting
     *
     * @param event               - created event or old state of updated\removed event
     * @param modificationInfoOpt - information about event modification
     * @param type                - action type
     * @return
     */
    private Optional<Instant> calculateEventTimeBeforeUpdate(EventAndRepetition event, Optional<ModificationInfo> modificationInfoOpt, XivaNotificationType type) {
        switch (type) {
            case UPDATED:
                if (modificationInfoOpt.isEmpty()) {
                    return Optional.empty();
                }
                ModificationInfo modificationInfo = modificationInfoOpt.get();

                switch (modificationInfo.getScope()) {
                    case ALL:
                        return Optional.of(event.getClosestIntervalWithRecurrence(Instant.now()).get().getStart());
                    case SINGLE:
                        if (modificationInfo.getNewEvent().isPresent()) {
                            EventAndRepetition newEvent = modificationInfo.getNewEvent().get();
                            return newEvent.getEvent().getRecurrenceId().toOptional();
                        }

                        return Optional.of(event.getClosestIntervalWithRecurrence(Instant.now()).get().getStart());
                    default:
                        return Optional.empty();
                }
            case DELETED:
                return Optional.of(event.getClosestIntervalWithRecurrence(Instant.now()).get().getStart());
            default:
                return Optional.empty();
        }
    }

    private Optional<NameI18n> loadOrganizerName(EventAndRepetition eventAndRepetition) {
        Option<EventInstanceInterval> closestInterval = eventAndRepetition.getClosestIntervalWithRecurrence(Instant.now());

        long currentEventId;
        if (closestInterval.isPresent()) {
            Instant instanceStart = closestInterval.get().getEventInterval().getInstanceStart();
            ListF<Event> events = eventDao.findEventByMainIdAndRecurrenceId(
                    eventAndRepetition.getEvent().getMainEventId(),
                    new RecurrenceIdOrMainEvent(Option.of(instanceStart)),
                    false
            );

            currentEventId = events
                    .firstO()
                    .orElse(eventAndRepetition.getEvent())
                    .getId();
        } else {
            currentEventId = eventAndRepetition.getEventId();
        }

        Option<PassportUid> organizerUid = eventUsers.findOrganizerByEventId(currentEventId);

        if (organizerUid.isEmpty()) {
            return Optional.empty();
        }

        return userManager.getYtUserNameByUid(organizerUid.get());
    }

    private int calculateTtl(Instant startTs, XivaNotificationType type) {
        if (type == XivaNotificationType.DELETED) {
            return 24 * 60 * 60;
        }

        long ttlInMillis = startTs.minus(Instant.now().getMillis()).getMillis();
        return (int) ttlInMillis / 1000;
    }
}
