package ru.yandex.calendar.logic.notification.xiva;

import java.util.Objects;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author Stepan Kladikov
 */
public class XivaNotificationNeedSendChecker {
    static final int DEFAULT_ALLOWABLE_OFFSET_IN_DAYS = 3;

    public boolean isNotificationNeedSend(Option<PassportUid> actorUid,
                                          PassportUid destinationUid,
                                          NotificationPreparedData data) {
        if (actorUid.isNotEmpty()
                && Objects.equals(actorUid.get(), destinationUid)) {
            return false;
        }

        if (!destinationUid.isYandexTeamRu()) {
            return false;
        }

        if (data.getEventType().isAbsence()) {
            return false;
        }

        switch (data.getType()) {
            case MEETING_REMINDER:
            case NOT_MEETING_REMINDER:
            case CREATED:
            case DELETED:
                return data.getCurrentStartTs().isAfterNow();
            case UPDATED:
                if (data.getOldStartTs().isEmpty()) {
                    return false;
                }
                return isEventTimeChangedInAllowableRange(data.getCurrentStartTs(), data.getOldStartTs().get());
            default:
                return false;
        }
    }

    private boolean isEventTimeChangedInAllowableRange(Instant newEventStartTs, Instant oldEventStartTs) {
        if (oldEventStartTs.isEqual(newEventStartTs)) {
            return false;
        }

        InstantInterval thresholdInterval = new InstantInterval(
                Instant.now(),
                Duration.standardDays(XivaNotificationNeedSendChecker.DEFAULT_ALLOWABLE_OFFSET_IN_DAYS)
        );

        return thresholdInterval.contains(newEventStartTs) || thresholdInterval.contains(oldEventStartTs);
    }
}
