package ru.yandex.calendar.logic.notification.xiva.content;

import java.util.Optional;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.xiva.v2.models.XivaEntity;
import ru.yandex.calendar.frontend.xiva.v2.models.payload.XivaPayload;
import ru.yandex.calendar.frontend.xiva.v2.models.repack.XivaRepack;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.NameI18n;

/**
 * @author Stepan Kladikov
 */
public class XivaNotificationEntityBuilder {
    private final String service = "calendar";
    private final String baseCalendarUrl = "https://calendar.yandex-team.ru";

    public XivaEntity build(
            long eventId,
            long uid,
            String eventName,
            Instant startTs,
            Instant oldStartTs,
            XivaNotificationType type,
            Settings settings,
            Optional<NameI18n> organizer,
            ListF<ResourceInfo> resources
    ) {
        String url = buildUrl(eventId);
        String subtitle = buildSubtitle(type, settings.getLanguage());
        String androidTitle = subtitle + ": " + eventName;
        String iosTitle = eventName;
        String description = buildDescription(
                startTs,
                oldStartTs,
                type,
                settings.getLanguage(),
                DateTimeZone.forID(settings.getTimezoneJavaid()),
                organizer,
                resources
        );

        XivaPayload payload = new XivaPayload(service, androidTitle, description, url, uid);
        XivaRepack repack = new XivaRepack(androidTitle, iosTitle, subtitle, description);
        return new XivaEntity(payload, repack, Cf.list(service));
    }

    private String buildUrl(Long eventId) {
        return baseCalendarUrl + "/event/" + eventId;
    }

    private String buildSubtitle(XivaNotificationType type, Language lang) {
        switch (type) {
            case CREATED:
                return TemporaryLocalizedStrings.CREATED_EVENT_SUBTITLE.getName(lang);
            case UPDATED:
                return TemporaryLocalizedStrings.UPDATED_EVENT_SUBTITLE.getName(lang);
            case DELETED:
                return TemporaryLocalizedStrings.DELETED_EVENT_SUBTITLE.getName(lang);
            case MEETING_REMINDER:
            case NOT_MEETING_REMINDER:
                return TemporaryLocalizedStrings.STARTING_SOON_SUBTITLE.getName(lang);
            default:
                throw new IllegalArgumentException("Unrecognized XivaNotificationType: " + type.name());
        }
    }

    private String buildDescription(
            Instant startTs,
            Instant oldStartTs,
            XivaNotificationType type,
            Language lang,
            DateTimeZone zoneId,
            Optional<NameI18n> organizer,
            ListF<ResourceInfo> resources
    ) {
        StringBuilder builder = new StringBuilder();

        switch (type) {
            case DELETED:
            case CREATED:
            case NOT_MEETING_REMINDER:
                builder.append(TemporaryLocalizedStrings.WHEN.getName(lang))
                        .append(": ")
                        .append(formatDate(startTs, zoneId, lang))
                        .append("\n");
                break;
            case UPDATED:
                builder.append(TemporaryLocalizedStrings.OLD_DATE.getName(lang))
                        .append(": ")
                        .append(formatDate(oldStartTs, zoneId, lang))
                        .append("\n")
                        .append(TemporaryLocalizedStrings.NEW_DATE.getName(lang))
                        .append(": ")
                        .append(formatDate(startTs, zoneId, lang))
                        .append("\n");

                break;
            case MEETING_REMINDER:
                builder.append(TemporaryLocalizedStrings.WHEN.getName(lang))
                        .append(": ")
                        .append(formatDate(startTs, zoneId, lang))
                        .append("\n");

                ListF<String> formattedResources = resources
                        .map(resource -> formatResource(resource, lang))
                        .filter(str -> !str.isBlank());

                if (formattedResources.isNotEmpty()) {
                    builder.append(TemporaryLocalizedStrings.RESOURCES.getName(lang))
                            .append(": ");
                    formattedResources.forEach(str -> builder.append(str).append(" "));
                    builder.append("\n");
                }
                break;
            default:
                throw new IllegalArgumentException("Unrecognized XivaNotificationType: " + type.name());
        }

        organizer.ifPresent(nameI18n ->
                builder
                        .append(TemporaryLocalizedStrings.ORGANIZER.getName(lang))
                        .append(": ")
                        .append(nameI18n.getName(lang)));

        return builder.toString();
    }

    private String formatDate(Instant timeTs, DateTimeZone tz, Language lang) {
        StringBuilder builder = new StringBuilder();
        LocalDate eventDate = new LocalDate(timeTs, tz);
        LocalDate today = LocalDate.now(tz);
        LocalDate yesterday = today.minusDays(1);
        LocalDate tomorrow = today.plusDays(1);

        if (yesterday.isEqual(eventDate)) {
            builder.append(TemporaryLocalizedStrings.YESTERDAY.getName(lang));
        } else if (today.isEqual(eventDate)) {
            builder.append(TemporaryLocalizedStrings.TODAY.getName(lang));
        } else if (tomorrow.isEqual(eventDate)) {
            builder.append(TemporaryLocalizedStrings.TOMORROW.getName(lang));
        } else {
            builder.append(TemporaryLocalizedStrings.dayOfWeek(timeTs.getMillis(), tz.getID(), lang));
            builder.append(", ");
            builder.append(TemporaryLocalizedStrings.date(timeTs.getMillis(), tz.getID(), lang));
        }

        builder.append(", ");
        builder.append(TemporaryLocalizedStrings.AT_TIME.getName(lang));
        builder.append(" ");
        builder.append(TemporaryLocalizedStrings.time(timeTs.getMillis(), tz.getID(), lang));

        return builder.toString();
    }

    private String formatResource(ResourceInfo resource, Language lang) {
        StringBuilder builder = new StringBuilder();
        String resourceName = resource.getNameI18n(lang)
                .orElse(TemporaryLocalizedStrings.UNKNOWN_ROOM.getName(lang));
        Option<Integer> floorNumber = resource.getFloorNum();

        if (resourceName.isEmpty()) {
            return "";
        }

        builder.append(resourceName);

        if (floorNumber.isPresent()) {
            builder.append(" (")
                    .append(floorNumber.get())
                    .append(" ")
                    .append(TemporaryLocalizedStrings.FLOOR.getName(lang))
                    .append(")");
        }

        return builder.toString();
    }
}
