package ru.yandex.calendar.logic.notification.xiva.notify;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.xiva.v2.XivaV2;
import ru.yandex.calendar.frontend.xiva.v2.models.XivaEntity;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.event.ModificationInfo;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.notification.xiva.NotificationPreparedData;
import ru.yandex.calendar.logic.notification.xiva.NotificationPreparedDataBuilder;
import ru.yandex.calendar.logic.notification.xiva.XivaNotificationNeedSendChecker;
import ru.yandex.calendar.logic.notification.xiva.XivaSendNotificationExecutor;
import ru.yandex.calendar.logic.notification.xiva.content.XivaNotificationEntityBuilder;
import ru.yandex.calendar.logic.notification.xiva.content.XivaNotificationType;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class XivaCommonManager {
    private static final Logger logger = LoggerFactory.getLogger(XivaCommonManager.class);

    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private NotificationPreparedDataBuilder preparedDataBuilder;
    @Autowired
    private XivaNotificationNeedSendChecker needSendChecker;
    @Autowired
    private XivaNotificationEntityBuilder payloadBuilder;
    @Autowired
    private XivaSendNotificationExecutor executor;
    @Autowired
    private XivaV2 xivaV2;

    public void handle(
            EventAndRepetition originEvent,
            ListF<PassportUid> uids,
            Option<PassportUid> actorUid,
            XivaNotificationType type
    ) {
        handle(originEvent, Optional.empty(), uids, actorUid, type);
    }

    public void handle(
            EventAndRepetition originEvent,
            Optional<ModificationInfo> modificationInfo,
            ListF<PassportUid> uids,
            Option<PassportUid> actorUid,
            XivaNotificationType type
    ) {
        handle(originEvent, modificationInfo, uids, actorUid, Cf.list(), type, true);
    }

    public void handleSynchronously(
            EventAndRepetition originEvent,
            ListF<PassportUid> uids,
            Option<PassportUid> actorUid,
            ListF<ResourceInfo> resources,
            XivaNotificationType type
    ) {
        handle(originEvent, Optional.empty(), uids, actorUid, resources, type, false);
    }

    private void handle(
            EventAndRepetition originEvent,
            Optional<ModificationInfo> modificationInfo,
            ListF<PassportUid> uids,
            Option<PassportUid> actorUid,
            ListF<ResourceInfo> resources,
            XivaNotificationType type,
            Boolean isAsync
    ) {
        MapF<PassportUid, SettingsInfo> settingToUids = settingsRoutines.getSettingsByUidBatch(uids);
        NotificationPreparedData data = preparedDataBuilder.build(originEvent, modificationInfo, type);

        ListF<PassportUid> uidsToSend = uids.filter(uid ->
                needSendChecker.isNotificationNeedSend(actorUid, uid, data));

        Consumer<ListF<PassportUid>> send = (uidsBatch) ->
                buildAndSendNotifications(uidsBatch, settingToUids, data, resources);

        logger.info("Sending xiva push notification caused for eventId {}, isAsync {}", data.getEventId(), isAsync);
        if (isAsync) {
            executor.schedule(uidsToSend, send);
        } else {
            executor.execute(uidsToSend, send);
        }
        logger.info("Sending xiva push notification for eventId({}) finished.", data.getEventId());
    }

    private void buildAndSendNotifications(
            ListF<PassportUid> uids,
            MapF<PassportUid, SettingsInfo> settingToUids,
            NotificationPreparedData data,
            ListF<ResourceInfo> resources
    ) {
        uids.forEach(uid -> {
            SettingsInfo userSettings = settingToUids.getOrThrow(uid);
            ListF<ResourceInfo> currentResources = calculateUserResources(resources, userSettings.getYt());

            XivaEntity entity = payloadBuilder.build(
                    data.getEventId(),
                    uid.getUid(),
                    data.getEventName(),
                    data.getCurrentStartTs(),
                    data.getOldStartTs()
                            .orElse(data.getCurrentStartTs()),
                    data.getType(),
                    userSettings.getCommon(),
                    data.getOrganizer(),
                    currentResources
            );

            xivaV2.push(uid, entity, data.getType(), data.getTtl());
        });
    }

    private ListF<ResourceInfo> calculateUserResources(ListF<ResourceInfo> resources, Option<SettingsYt> settingsYt) {
        if (settingsYt.isEmpty()) {
            return Cf.list();
        }

        Option<Long> officeId = settingsYt.get().getActiveOfficeId();
        if (officeId.isEmpty()) {
            return Cf.list();
        }

        return resources.filter(resource ->
                Objects.equals(resource.getOffice().getId(), officeId.get())
        );
    }
}
