package ru.yandex.calendar.logic.resource;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Validate;

/**
 * @author yashunsky
 */
public class MultiofficeResourceFilter {
    private final ResourceFilter commonFilter;
    private final ListF<Long> officeIds;
    private final MapF<Long, ResourceFilter> officeSpecificFilters;

    public MultiofficeResourceFilter(
            ListF<Long> officeIds, ListF<String> filters, Option<String> resourceNamePrefix, ListF<Email> emails)
    {
        Validate.in(filters.size(), Cf.list(0, 1, officeIds.size()));

        ResourceFilter resourceFilter = ResourceFilter.any();

        if (resourceNamePrefix.isPresent() && !resourceNamePrefix.get().isEmpty()) {
            resourceFilter = resourceFilter.withPrefix(resourceNamePrefix.get());
        }

        if (emails.isNotEmpty()) {
            resourceFilter = resourceFilter.withEmailFilter(emails);
        }

        if (filters.size() == 1) {
            resourceFilter = resourceFilter.withFilterSafe(filters.single());
        }
        this.officeIds = officeIds;
        this.commonFilter = resourceFilter;
        this.officeSpecificFilters = filters.size() > 1
                ? officeIds.zip(filters).toMap().mapValues(commonFilter::withFilterSafe) : Cf.map();
    }

    public boolean isOfficeSpecific() {
        return officeSpecificFilters.isNotEmpty();
    }

    public ListF<Long> getOfficeIds() {
        return officeIds;
    }

    public ResourceFilter getFilter() {
        Validate.isFalse(isOfficeSpecific());
        return commonFilter;
    }

    public Tuple2List<Long, ResourceFilter> getFiltersByOffice() {
        return (isOfficeSpecific() ? officeSpecificFilters : officeIds.zipWith(id -> commonFilter).toMap()).entries();
    }

    public Option<String> getResourceNamePrefix() {
        return commonFilter.getResourceNamePrefix();
    }

    public Function1B<Resource> nameWordMatchesPrefixF() {
        return commonFilter.nameWordMatchesPrefixF();
    }
}
