package ru.yandex.calendar.logic.resource;

import javax.annotation.PreDestroy;

import lombok.val;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.beans.generated.Office;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.resources.UStringLiteral;
import ru.yandex.inside.geobase.GeobaseIds;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportDomain;
import ru.yandex.misc.cache.NoParameterCache;
import ru.yandex.misc.cache.impl.SimpleNoParameterCache;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

public class OfficeManager {
    public static final DateTimeZone MOSCOW_TIMEZONE = DateTimeZone.forID("Europe/Moscow");

    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private UserManager userManager;
    @Autowired
    private EnvironmentType environmentType;


    public static final ListF<Integer> KR_OFFICE_CENTER_IDS = Cf.list(1, 147, 148, 173);
    public static final int AURORA_CENTER_ID = 181;
    public static final int MOROZOV_CENTER_ID = 1;

    public DateTimeZone getTimeZoneByOfficeId(long officeId) {
        return getOfficeTimeZone(resourceDao.findOfficeById(officeId).getOrThrow("no office with id " + officeId));
    }

    public static DateTimeZone getOfficeTimeZone(Office office) {
        return office.getTimezoneId().isPresent() ? DateTimeZone.forID(office.getTimezoneId().get()) : MOSCOW_TIMEZONE;
    }

    public static int getCountryIdByCityName(String cityName) {
        if (StringUtils.isEmpty(cityName)) {
            return GeobaseIds.RUSSIA;
        }
        cityName = cityName.toLowerCase();

        if (cityName.contains(UStringLiteral.KIEV.toLowerCase())
            || cityName.contains(UStringLiteral.ODESSA.toLowerCase()))
        {
            return GeobaseIds.UKRAINE;
        }
        if (cityName.contains(UStringLiteral.MINSK.toLowerCase())) {
            return GeobaseIds.BELARUS;
        }
        if (cityName.contains(UStringLiteral.ISTANBUL.toLowerCase())) {
            return GeobaseIds.TURKEY;
        }
        if (cityName.contains(UStringLiteral.CALIFORNIA.toLowerCase())) {
            return GeobaseIds.USA;
        }
        if (cityName.contains(UStringLiteral.ZURICH.toLowerCase())) {
            return GeobaseIds.ZURICH;
        }
        if (cityName.contains(UStringLiteral.LUCERNE.toLowerCase())) {
            return GeobaseIds.LUCERNE;
        }
        if (cityName.contains(UStringLiteral.ALMATY.toLowerCase())) {
            return GeobaseIds.KAZAKHSTAN;
        }
        if (cityName.contains(UStringLiteral.TEL_AVIV.toLowerCase())) {
            return GeobaseIds.ISRAEL;
        }

        if (Cf.list(
                UStringLiteral.BERLIN.toLowerCase(),
                UStringLiteral.GERMANY.toLowerCase()).exists(cityName::contains))
        {
            return GeobaseIds.GERMANY;
        }
        if (cityName.contains(UStringLiteral.KAZAN.toLowerCase())) {
            return GeobaseIds.TATARSTAN;
        }
        if (cityName.contains(UStringLiteral.SIMFEROPOL.toLowerCase())) {
            return GeobaseIds.CRIMEA;
        }
        if (cityName.contains(UStringLiteral.PRAGUE.toLowerCase())) {
            return GeobaseIds.CZECH_REPUBLIC;
        }

        return GeobaseIds.RUSSIA;
    }

    private final NoParameterCache<ListF<Office>> officesCache =
            new SimpleNoParameterCache<>(() -> resourceDao.findOfficesInAllDomains(), Duration.standardMinutes(10));

    @PreDestroy
    public void destroy() {
        officesCache.destroy();
    }

    private ListF<Office> getOffices() {
        return environmentType != EnvironmentType.TESTS ? officesCache.get() : resourceDao.findOfficesInAllDomains();
    }

    public int getCountryIdByOfficeId(long officeId) {
        return getCountryIdByCityName(getOfficesByIds(Cf.list(officeId)).single().getCityName().getOrElse(""));
    }

    public Tuple2List<Integer, Option<Office>> getOfficesByCenterIds(ListF<Integer> centerIds) {
        val found = getOffices().find(Office.getCenterIdF().andThen(Cf2.isSomeOfF(centerIds.unique())));

        return centerIds.zipWith(
                found.toMapMappingToKey(Office.getCenterIdF().andThen(Cf2.f(Option::get)))::getO);
    }

    public Option<Office> getOfficeByCenterId(int centerId) {
        return getOfficesByCenterIds(Option.of(centerId)).single().get2();
    }

    public Option<Long> getOfficeIdByCenterId(int centerId) {
        return getOfficeByCenterId(centerId).map(Office.getIdF());
    }

    public Office getOfficeById(long officeId) {
        return getOfficesByIds(Cf.list(officeId)).single();
    }

    public ListF<Office> getOfficesByIds(ListF<Long> officeIds) {
        val idsUnique = officeIds.unique();
        val found = getOffices().filter(Office.getIdF().andThen(idsUnique.containsF()));

        Validate.sameSize(idsUnique, found, "offices not found by some of ids " + idsUnique);
        return found;
    }

    public ListF<Office> getOfficesByStaffIds(ListF<Long> officeIds) {
        val idsUnique = officeIds.unique();
        val found = getOffices()
                .filter(office -> office.getStaffId().exists(idsUnique::containsTs));

        Validate.sameSize(idsUnique, found, "offices not found by some of ids " + idsUnique);
        return found;
    }

    public ListF<Office> getDomainOfficesWithAtLeastOneActiveResource(PassportUid uid, PassportDomain domain) {
        val user = userManager.getUserInfo(uid);

        val resourceCondition = ResourceFields.IS_ACTIVE.eq(Boolean.TRUE)
                .and(ResourceFields.TYPE.column().inSet(user.getResourceTypesCanView()));

        return resourceDao.findOfficesByResources(domain, resourceCondition);
    }

    public ListF<Office> getDomainOfficesWithAtLeaseOneResourceWithType(
            PassportUid uid, PassportDomain domain, SetF<ResourceType> types)
    {
        val user = userManager.getUserInfo(uid);

        val resourceCondition = ResourceFields.IS_ACTIVE.eq(Boolean.TRUE)
                .and(ResourceFields.TYPE.column().inSet(user.getResourceTypesCanView().intersect(types)));

        return resourceDao.findOfficesByResources(domain, resourceCondition);
    }

    public Option<Office> getDefaultOfficeForUser(PassportUid uid) {
        return resourceDao.findOfficeForUser(uid);
    }

    public Option<Long> chooseActiveOfficeId(Option<Long> lastActivityOfficeId, Option<Long> tableOfficeId, PassportUid uid) {
        return lastActivityOfficeId
            .map(id -> chooseActiveOfficeId(id, tableOfficeId, uid))
            .orElse(() -> userManager.isYamoneyUser(uid) ? tableOfficeId : Option.empty());
    }

    public Long chooseActiveOfficeId(Long lastActivityOfficeId, Option<Long> tableOfficeId, PassportUid uid) {
        Function1B<Long> isKrOfficeF = id -> getOfficeById(id).getCenterId().exists(KR_OFFICE_CENTER_IDS.containsF());
        Function1B<Long> isAuroraOfficeF = id -> getOfficeById(id).getCenterId().isSome(AURORA_CENTER_ID);

        if (isKrOfficeF.apply(lastActivityOfficeId)) {
            return tableOfficeId.exists(isKrOfficeF.or(isAuroraOfficeF))
                    ? tableOfficeId.get()
                    : getOfficeByCenterId(MOROZOV_CENTER_ID).map(Office::getId).getOrElse(lastActivityOfficeId);
        } else {
            return lastActivityOfficeId;
        }
    }

    public static Function<Office, DateTimeZone> getOfficeTimeZoneF() {
        return OfficeManager::getOfficeTimeZone;
    }

    public Function<Long, DateTimeZone> getTimeZoneByOfficeIdF() {
        return this::getTimeZoneByOfficeId;
    }

    public Function<Integer, Option<Long>> getOfficeIdByCenterIdF() {
        return this::getOfficeIdByCenterId;
    }

    public Function<Long, Integer> getCountryIdByOfficeIdF() {
        return this::getCountryIdByOfficeId;
    }
}
