package ru.yandex.calendar.logic.resource;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.calendar.logic.user.NameI18n;

/**
 * @author yashunsky
 */
public class RejectedResources {
    public enum Reason {
        INACCESSIBLE(". из-за непреемлимых для этой переговорки параметров встречи.",
                "  because of meeting parameters unacceptable for this room."),
        NOT_PERMITTED(". У вас нет права бронировать эту переговорку.",
                ". You don't have the permission to book this room."),
        BUSY(" из-за наличия конфликтов.", " because of booking conflicts.");

        private final NameI18n exchangeReason;

        private static final String COMMON_RU = "Ваше приглашение было отклонено";
        private static final String COMMON_EN = "Your invitation has been declined";

        Reason(String ru, String en) {
            this.exchangeReason = new NameI18n(COMMON_RU + ru, COMMON_EN + en);
        }

        public NameI18n getExchangeReason() {
            return exchangeReason;
        }
    }

    public static final RejectedResources EMPTY = new RejectedResources(Cf.map());

    private final MapF<Reason, ListF<Long>> resources;

    public RejectedResources(MapF<Reason, ListF<Long>> resources) {
        this.resources = resources;
    }

    public static RejectedResources inaccessible(ListF<Long> resources) {
        return new RejectedResources(Cf.map(Reason.INACCESSIBLE, resources));
    }

    public static RejectedResources notPermitted(ListF<Long> resources) {
        return new RejectedResources(Cf.map(Reason.NOT_PERMITTED, resources));
    }

    public static RejectedResources busy(ListF<Long> resources) {
        return new RejectedResources(Cf.map(Reason.BUSY, resources));
    }

    public RejectedResources add(Reason reason, ListF<Long> ids) {
        return new RejectedResources(resources.plus(
                Cf.map(reason, resources.getOrElse(reason, Cf.list()).plus(ids).stableUnique())));
    }

    public RejectedResources add(RejectedResources rejectedResources) {
        return new RejectedResources(Cf.x(Reason.values()).filterMap(reason -> {
            ListF<Long> ids = this.get(reason).plus(rejectedResources.get(reason)).stableUnique();
            return Option.when(ids.isNotEmpty(), Tuple2.tuple(reason, ids));
        }).toMap(t -> t._1, t -> t._2));
    }

    public ListF<Long> getAllIds() {
        return resources.values().flatten();
    }

    public ListF<Long> get(Reason reason) {
        return resources.getOrElse(reason, Cf.list());
    }

    public Option<NameI18n> getReason(long resourceId) {
        return resources.filterValues(ids -> ids.containsTs(resourceId)).keys().firstO().map(Reason::getExchangeReason);
    }

    public boolean isNotEmpty() {
        return resources.isNotEmpty();
    }
}
