package ru.yandex.calendar.logic.resource;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function0;
import ru.yandex.calendar.frontend.web.cmd.run.PermissionDeniedUserException;
import ru.yandex.calendar.frontend.web.cmd.run.ReadableErrorMessageException;
import ru.yandex.calendar.frontend.web.cmd.run.ResourceBusyOverlapException;
import ru.yandex.calendar.logic.beans.generated.EventResource;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActorId;
import ru.yandex.calendar.logic.event.EventChangesInfo;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.InaccessibleResourcesRequest;
import ru.yandex.calendar.logic.event.avail.AvailabilityOverlap;
import ru.yandex.calendar.logic.event.dao.EventResourceDao;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.resource.reservation.ResourceReservationManager;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.StringUtils;

@Slf4j
public class ResourceAccessRoutines {
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private ResourceReservationManager resourceReservationManager;
    @Autowired
    private UserManager userManager;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private EventResourceDao eventResourceDao;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private Authorizer authorizer;

    public static boolean throwOnFailure(ActionInfo actionInfo) {
        return !actionInfo.getActionSource().isFromExchangeOrMail();
    }

    public RejectedResources checkForBusyResources(ActorId occupier, EventWithRelations event,
            RepetitionInstanceInfo repetitionInfo, ListF<Long> exceptEventIds, ActionInfo actionInfo)
    {
        return checkForBusyResources(
                occupier, event, repetitionInfo, exceptEventIds, throwOnFailure(actionInfo), actionInfo);
    }

    public RejectedResources checkForBusyResources(ActorId occupier, EventWithRelations event,
            RepetitionInstanceInfo repetitionInfo, ListF<Long> exceptEventIds,
            boolean throwExceptionOnFailure, ActionInfo actionInfo)
    {
        if (occupier.isUser()) {
            resourceReservationManager.cancelReservations(occupier.getUid(), event);
        }

        Tuple2List<Long, AvailabilityOverlap> overlaps = eventRoutines.findResourcesEventsIntersectingGivenInterval(
                occupier.getUidO(), event.getResourceIds(), exceptEventIds.plus(event.getId()).stableUnique(),
                repetitionInfo, false, throwExceptionOnFailure, actionInfo);

        if (overlaps.isEmpty()) {
            return RejectedResources.EMPTY;
        }

        if (throwExceptionOnFailure) {
            Tuple2<Long, AvailabilityOverlap> overlap = overlaps.first();
            throw new ResourceBusyOverlapException("Resource layers deny busy overlaps."
                    + " Resource id: " + overlap.get1()
                    + ", event id: " + event.getEvent().getId()
                    + ", external id: " + event.getMainEvent().getExternalId()
                    + ", first intersecting event: " + overlap.get2().getEventIdOrReservation()
                    + ", overlap interval: " + overlap.get2().getOverlap(),
                    event.getResources().find(r -> r.getResourceId() == overlap.get1()).get().getResource(),
                    overlap.get2());
        } else {
            for (Tuple2<Long, AvailabilityOverlap> overlap : overlaps) {
                log.info("Deleting resource due to availability overlap."
                                + " Resource: {}, event: {}, first intersecting event: {}, overlap interval: {}",
                        overlap.get1(), event.getId(),
                        overlap.get2().getEventIdOrReservation(), overlap.get2().getOverlap());
            }

            ListF<Long> resourceIds = overlaps.get1();
            storeAndDeleteEventResources(event.getEventResources(), resourceIds, actionInfo);
            return RejectedResources.busy(resourceIds);
        }
    }

    public RejectedResources checkBookingPermissions(
            ActorId subjectId, long eventId, ListF<Long> newResourceIds, ActionInfo actionInfo)
    {
        return checkBookingPermissions(subjectId, eventId, newResourceIds, throwOnFailure(actionInfo), actionInfo);
    }

    public RejectedResources checkBookingPermissions(
            ActorId subjectId, long eventId, ListF<Long> newResourceIds,
            boolean throwExceptionOnFailure, ActionInfo actionInfo)
    {
        val userO = resolveUser(subjectId, () -> eventUserDao.findOrganizerByEventId(eventId), actionInfo);

        if (!userO.isPresent()) {
            return RejectedResources.notPermitted(newResourceIds);
        }

        val user = userO.get();
        val resources = resourceRoutines.getResourcesByIds(newResourceIds);
        val notPermittedIds = Cf.toList(authorizer.getNonBookableResourceIds(user, resources));

        if (notPermittedIds.isEmpty()) {
            return RejectedResources.EMPTY;
        }

        if (throwExceptionOnFailure) {
            throw new PermissionDeniedUserException(
                    "Not allowed to invite some of resources " + notPermittedIds);
        } else {
            notPermittedIds.forEach(id -> log.info("Deleting resource not allowed to be booked."
                    + " Resource: {}, event: {}, reason: {}, user: {}", id, eventId, user.getUid()));

            storeAndDeleteEventResources(
                    eventResourceDao.findEventResourcesByEventId(eventId), notPermittedIds, actionInfo);
            return RejectedResources.notPermitted(notPermittedIds);
        }
    }

    public RejectedResources checkForInaccessibleResources(
            ActorId occupier, EventWithRelations event,
            RepetitionInstanceInfo repetitionInfo, InaccessibilityCheck inaccessibilityCheck, ActionInfo actionInfo)
    {
        return checkForInaccessibleResources(
                occupier, event, repetitionInfo, inaccessibilityCheck, throwOnFailure(actionInfo), actionInfo);
    }

    public RejectedResources checkForInaccessibleResources(ActorId occupier, EventWithRelations event,
            RepetitionInstanceInfo repetitionInfo, InaccessibilityCheck inaccessibilityCheck,
            boolean throwExceptionOnFailure, ActionInfo actionInfo)
    {

        ListF<ResourceInfo> resourcesToCheck = inaccessibilityCheck.getResourcesToCheck(event);

        if (resourcesToCheck.isEmpty()) {
            return RejectedResources.EMPTY;
        }

        Option<UserInfo> userInfo = resolveUser(occupier, event, actionInfo);
        ListF<ResourceInaccessibility> inaccessible = eventRoutines.findInaccessibleResources(
                userInfo,
                InaccessibleResourcesRequest.verify(
                        resourcesToCheck, event.getParticipants().getParticipantUidsSafeWithInconsistent(),
                        repetitionInfo, Option.of(event.getId()), actionInfo.getNow()
                )
        );

        if (inaccessible.isEmpty()) {
            if (throwExceptionOnFailure && StringUtils.isBlank(event.getEvent().getDescription())
                    && resourcesToCheck.iterator().map(ResourceInfo::getExchangeName)
                            .exists(SpecialResources.guestsDescriptionRequiredRooms.get()::containsTs))
            {
                throw new ReadableErrorMessageException(new NameI18n(
                        "Пожалуйста, укажите в описании, кто гости и сколько их (нажмите «Добавить описание»)",
                        "Please specify in summary whom and how many guests are expected (click \"Add summary\")"));
            }
            return RejectedResources.EMPTY;
        }

        if (throwExceptionOnFailure) {
            throw new ResourceInaccessibleException(inaccessible.first());
        } else {
            inaccessible.forEach(ri -> log.info(
                    "Deleting resource due to inaccessibility. Resource: {}, event: {}, reason: {}",
                    ri.getResourceId(), event.getId(), ri.getReadableMessage().getName(Language.ENGLISH)));

            ListF<Long> resourceIds = inaccessible.map(ResourceInaccessibility::getResourceId);
            storeAndDeleteEventResources(event.getEventResources(), resourceIds, actionInfo);
            return RejectedResources.inaccessible(resourceIds);
        }
    }

    private void storeAndDeleteEventResources(
            ListF<EventResource> eventResources, ListF<Long> resourceIds, ActionInfo actionInfo)
    {
        eventRoutines.storeAndDeleteEventResources(eventResources
                .filter(resourceIds.containsF().compose(EventResource::getResourceId)), actionInfo);
    }

    private Option<UserInfo> resolveUser(ActorId subjectId, EventWithRelations event, ActionInfo actionInfo) {
        return resolveUser(subjectId, event::getOrganizerUidIfMeeting, actionInfo);
    }

    private Option<UserInfo> resolveUser(
            ActorId subjectId, Function0<Option<PassportUid>> getOrganizer, ActionInfo actionInfo)
    {
        return (actionInfo.getActionSource().isFromExchangeOrMail()
                ? getOrganizer.apply()
                : subjectId.getUidO()).map(userManager::getUserInfo);
    }

    public static class InaccessibilityCheck {
        private final Option<EventChangesInfo> changes;

        private InaccessibilityCheck(Option<EventChangesInfo> changes) {
            this.changes = changes;
        }

        public static InaccessibilityCheck onCreate() {
            return new InaccessibilityCheck(Option.empty());
        }

        public static InaccessibilityCheck onUpdate(EventChangesInfo changes) {
            return new InaccessibilityCheck(Option.of(changes));
        }

        private ListF<ResourceInfo> getResourcesToCheck(EventWithRelations event) {
            ListF<ResourceInfo> infos = event.getResources();
            if (!changes.isPresent() || changes.get().timeOrRepetitionChanges()) {
                return infos;
            }
            ListF<Long> newResourceIds = changes.get().getEventParticipantsChangesInfo().getNewResources();
            return infos.filter(info -> newResourceIds.containsTs(info.getResourceId()));
        }
    }
}
