package ru.yandex.calendar.logic.resource;

import java.util.NoSuchElementException;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.dao.IncorrectResultSizeDataAccessException;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.logic.beans.GenericBeanDao;
import ru.yandex.calendar.logic.beans.generated.EventResource;
import ru.yandex.calendar.logic.beans.generated.EventResourceFields;
import ru.yandex.calendar.logic.beans.generated.EventResourceHelper;
import ru.yandex.calendar.logic.beans.generated.Office;
import ru.yandex.calendar.logic.beans.generated.OfficeFields;
import ru.yandex.calendar.logic.beans.generated.OfficeHelper;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.calendar.logic.beans.generated.ResourceHelper;
import ru.yandex.calendar.logic.sharing.participant.ResourceParticipantInfo;
import ru.yandex.calendar.util.db.BeanRowMapper;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.calendar.util.db.SqlFunction;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportDomain;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlOrder;
import ru.yandex.misc.db.q.SqlQueryUtils;
import ru.yandex.misc.ip.InternetDomainName;
import ru.yandex.misc.lang.Validate;

/**
 * @author Stepan Koltsov
 */
public class ResourceDao extends CalendarJdbcDaoSupport {

    @Autowired
    private GenericBeanDao genericBeanDao;

    @RunWithRandomTest
    public ListF<ResourceInfo> findDomainResourcesWithLayersAndOffices(
            PassportDomain domain, SqlCondition officeCondition, SqlCondition resourceCondition)
    {
        MasterSlaveContextHolder.PolicyHandle handle = MasterSlaveContextHolder.push(MasterSlavePolicy.R_SM);
        try {
            Validate.isTrue(!domain.sameAs(PassportDomain.YANDEX_RU));

            BeanRowMapper<Resource> rRm = ResourceHelper.INSTANCE.offsetRowMapper(0);
            BeanRowMapper<Office> oRm = OfficeHelper.INSTANCE.offsetRowMapper(rRm.nextOffset());

            String sql = "SELECT " + rRm.columns("r.") + ", " + oRm.columns("o.") + " FROM resource r" +
                    " INNER JOIN office o ON r.office_id = o.id" +
                    " WHERE r.domain = ?" +
                    " " + resourceCondition.andSqlForTable("r") +
                    " " + officeCondition.andSqlForTable("o") +
                    " ORDER BY r.pos";

            ListF<?> args = Cf.list(domain.getDomain().getDomain(), resourceCondition.args(), officeCondition.args());

            if (skipQuery(officeCondition.and(resourceCondition), sql, args)) return Cf.list();

            return getJdbcTemplate().query2(sql, rRm, oRm, args).map(ResourceInfo.consF());
        } finally {
            handle.popSafely();
        }
    }

    // XXX: require domain parameter
    @RunWithRandomTest
    public ListF<Office> findOffices(PassportDomain domain) {
        Validate.isTrue(!domain.sameAs(PassportDomain.YANDEX_RU));

        String q = "SELECT * FROM office where domain = ?";
        return getJdbcTemplate().queryForList(q, Office.class, domain.getDomain().getDomain());
    }

    @RunWithRandomTest
    public ListF<Office> findOfficesByResources(PassportDomain domain, SqlCondition resourceCondition) {
        Validate.isTrue(!domain.sameAs(PassportDomain.YANDEX_RU));

        resourceCondition = resourceCondition.and(ResourceFields.DOMAIN.eq(domain.getDomain().getDomain()));

        String q = "SELECT DISTINCT o.* FROM office o" +
                " INNER JOIN resource r ON r.office_id = o.id" +
                " WHERE " + resourceCondition.sqlForTable("r");

        if (skipQuery(resourceCondition, q, resourceCondition.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, Office.class, resourceCondition.args());
    }

    @RunWithRandomTest
    public ListF<Office> findOfficesInAllDomains() {
        String q = "SELECT * FROM office";
        return getJdbcTemplate().queryForList(q, Office.class);
    }

    // XXX: require domain parameter
    @RunWithRandomTest
    public Tuple2List<Office, Integer> findOfficesWithActiveResourceCount(PassportDomain domain) {
        Validate.isTrue(!domain.sameAs(PassportDomain.YANDEX_RU));

        BeanRowMapper<Office> oRm = OfficeHelper.INSTANCE.offsetRowMapper(0);
        RowMapper<Integer> countRm = (rs, rowNum) -> rs.getInt(oRm.nextOffset() + 1);

        String q = "SELECT " + oRm.columns("o.") + ", COUNT(r.id) c" +
                " FROM office o" +
                " LEFT JOIN resource r ON r.office_id = o.id" +
                " WHERE (r.is_active = TRUE OR r.id IS NULL) AND o.domain = ?" +
                " GROUP BY(o.id)";
        return getJdbcTemplate().query2(q, oRm, countRm, domain.getDomain().getDomain());
    }

    // XXX: require domain parameter
    @RunWithRandomTest
    public Option<Office> findOfficeByAbbr(String abbr, PassportDomain domain) {
        String q = "SELECT * FROM office WHERE abbr = ? AND domain = ?";
        return getJdbcTemplate().queryForOption(q, Office.class, abbr, domain.getDomain().getDomain());
    }

    public Office saveOffice(Office template) {
        long id = genericBeanDao.insertBeanGetGeneratedKey(template);
        Office r = new Office();
        r.setFields(template);
        r.setId(id);
        return r;
    }

    @RunWithRandomTest
    public ListF<ResourceParticipantInfo> findResourceParticipants(CollectionF<Long> eventIds) {
        BeanRowMapper<Resource> rRm = ResourceHelper.INSTANCE.offsetRowMapper(0);
        BeanRowMapper<Office> oRm = OfficeHelper.INSTANCE.offsetRowMapper(rRm.nextOffset());
        BeanRowMapper<EventResource> erRm = EventResourceHelper.INSTANCE.offsetRowMapper(oRm.nextOffset());

        SqlCondition c = EventResourceFields.EVENT_ID.column().inSet(eventIds);
        String q = "SELECT " + Cf.list(rRm.columns("r."), oRm.columns("o."), erRm.columns("er.")).mkString(", ") +
                " FROM resource r" +
                " INNER JOIN office o ON r.office_id = o.id" +
                " INNER JOIN event_resource er ON r.id = er.resource_id" +
                " WHERE " + c.sqlForTable("er");

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().query3(q, rRm, oRm, erRm, c.args())
                .map(t -> new ResourceParticipantInfo(new ResourceInfo(t.get1(), t.get2()), t.get3()));
    }

    @RunWithRandomTest
    public Tuple2List<Long, ResourceType> findResourceTypesByIds(ListF<Long> resourceIds) {
        SqlCondition c = ResourceFields.ID.column().inSet(resourceIds);
        String q = "SELECT id, type FROM resource" + c.whereSql();

        RowMapper<Long> rm1 = (rs, num) -> rs.getLong(1);
        RowMapper<ResourceType> rm2 = (rs, num) -> ResourceType.R.fromValue(rs.getInt(2));

        if (skipQuery(c, q, c.args())) return Tuple2List.tuple2List();

        return getJdbcTemplate().query2(q, rm1, rm2, c.args());
    }

    @RunWithRandomTest
    public ListF<ResourceParticipantInfo> findResourceLayersWithEvent(long eventId) {
        return findResourceParticipants(Cf.list(eventId));
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public ResourceInfo findResourceInfoByResourceId(long resourceId) {
        try {
            return findResourceInfosByIds(Cf.list(resourceId)).singleO().get();
        } catch (NoSuchElementException e) {
            throw new EmptyResultDataAccessException(1);
        }
    }

    @RunWithRandomTest
    public ListF<ResourceInfo> findResourceInfosByIds(ListF<Long> resourceIds) {
        BeanRowMapper<Resource> rRm = ResourceHelper.INSTANCE.offsetRowMapper(0);
        BeanRowMapper<Office> oRm = OfficeHelper.INSTANCE.offsetRowMapper(rRm.nextOffset());

        String sql =
            "SELECT " + rRm.columns("r.") + ", " + oRm.columns("o.") +
            " FROM resource r" +
            " INNER JOIN office o ON r.office_id = o.id" +
            " WHERE r.id " + SqlQueryUtils.inSet(resourceIds);

        if (skipQuery(resourceIds, sql)) return Cf.list();

        return getJdbcTemplate().query2(sql, rRm, oRm).map(ResourceInfo.consF());
    }

    public ListF<ResourceInfo> findResourceInfosByExchangeNames(ListF<String> exchangeNames) {
        BeanRowMapper<Resource> rRm = ResourceHelper.INSTANCE.offsetRowMapper(0);
        BeanRowMapper<Office> oRm = OfficeHelper.INSTANCE.offsetRowMapper(rRm.nextOffset());

        String sql =
            "SELECT " + rRm.columns("r.") + ", " + oRm.columns("o.") +
            " FROM resource r" +
            " INNER JOIN office o ON r.office_id = o.id" +
            " WHERE r.exchange_name " + SqlQueryUtils.inSet(exchangeNames);

        if (skipQuery(exchangeNames, sql)) return Cf.list();

        ListF<ResourceInfo> resourceInfos = getJdbcTemplate().query2(sql, rRm, oRm).map(ResourceInfo.consF());

        if (resourceInfos.size() != exchangeNames.size()) {
            logger.error("Not all resources found by exchange names, required: " + exchangeNames + ", found: " + resourceInfos);
        }

        return resourceInfos;
    }

    @RunWithRandomTest
    public ListF<ResourceParticipantInfo> findYaTeamResourceLayersWithEvents(CollectionF<Long> eventIds) {
        return findResourceParticipants(eventIds).filter(
                ResourceParticipantInfo.resourceInfoF().andThen(ResourceInfo.isYaTeamF()));
    }

    @RunWithRandomTest
    public ListF<ResourceParticipantInfo> findYaTeamResourceLayersThatSyncWithExchangeWithEvents(
            CollectionF<Long> eventIds)
    {
        return findResourceParticipants(eventIds).filter(ResourceParticipantInfo.isYaTeamAndSyncWithExchangeF());
    }

    @RunWithRandomTest
    public ListF<ResourceParticipantInfo> findYaTeamResourceLayersThatSyncWithExchangeWithEvent(long eventId) {
        return findYaTeamResourceLayersThatSyncWithExchangeWithEvents(Cf.list(eventId));
    }

    @RunWithRandomTest
    public Option<Office> findOfficeForUser(PassportUid uid) {
        String q = "SELECT * FROM office WHERE id IN (SELECT active_office_id FROM settings_yt WHERE uid = ?)";
        return getJdbcTemplate().queryForOption(q, Office.class, uid);
    }

    public int deleteResourcesByDomain(PassportDomain domain) {
        String q = "DELETE FROM resource WHERE domain = ?";
        return getJdbcTemplate().update(q, domain.getDomain().getDomain());
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void deleteResourceById(long resourceId) {
        String q = "DELETE FROM resource WHERE id = ?";
        getJdbcTemplate().updateRow(q, resourceId);
    }

    @RunWithRandomTest(possible=DataIntegrityViolationException.class)
    public int deleteOfficesByDomain(PassportDomain domain) {
        String q = "DELETE FROM office WHERE domain = ?";
        return getJdbcTemplate().update(q, domain.getDomain().getDomain());
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public Resource findResourceById(long id) {
        return findResource(ResourceFields.ID.eq(id));
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public Resource findResourceByExchangeName(String exchangeName) {
        return findResource(ResourceFields.EXCHANGE_NAME.eq(exchangeName));
    }

    @RunWithRandomTest
    public ListF<Resource> findResourcesByIds(ListF<Long> ids) {
        return findResources(ResourceFields.ID.column().inSet(ids));
    }

    @RunWithRandomTest
    public ListF<Long> findResourceIdsAsyncWithExchangeByIds(ListF<Long> ids) {
        SqlCondition c = ResourceFields.ASYNC_WITH_EXCHANGE.eq(true).and(ResourceFields.ID.column().inSet(ids));
        String q = "SELECT id FROM resource" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class, c.args());
    }

    @RunWithRandomTest
    public Option<Resource> findResourceByDisplayToken(String displayToken) {
        return findResources(ResourceFields.DISPLAY_TOKEN.eq(displayToken)).singleO();
    }

    @RunWithRandomTest
    public ListF<Long> findAllYaTeamActiveResourceIdsThatSyncWithExchange() {
        String sql = "SELECT id FROM resource WHERE domain = ? AND is_active = TRUE AND sync_with_exchange = TRUE";
        return genericBeanDao.getJdbcTemplate().queryForList(sql, Long.class, ResourceRoutines.YT_DOMAIN.getDomain());
    }

    @RunWithRandomTest
    public Option<Resource> findResourceByDomainAndExchangeName(InternetDomainName domain, String exchangeName) {
        return findResourcesByDomainAndExchangeNames(domain, Cf.list(exchangeName)).get2().singleO();
    }

    @RunWithRandomTest
    public Tuple2List<String, Resource> findResourcesByDomainAndExchangeNames(
            InternetDomainName domain, ListF<String> names)
    {
        MapF<String, String> nameByLowerCaseName = names.toMap(String::toLowerCase, name -> name);

        return findResources(SqlFunction.lower(ResourceFields.EXCHANGE_NAME.column()).inSet(nameByLowerCaseName.keys())
                .and(ResourceFields.DOMAIN.eq(domain.getDomain())))
                .zipWith(r -> r.getExchangeName().filterMap(nameByLowerCaseName::getO))
                .filterBy2(Option::isPresent).map2(Option::get).invert();
    }

    @RunWithRandomTest
    public ListF<Resource> findResourcesByOfficeId(long officeId) {
        return findResources(ResourceFields.OFFICE_ID.eq(officeId));
    }

    @RunWithRandomTest
    public ListF<Resource> findResourcesSorted(SqlOrder o) {
        return findResourcesSorted(SqlCondition.trueCondition(), o);
    }

    @RunWithRandomTest
    public ListF<Resource> findResources() {
        return findResources(SqlCondition.trueCondition());
    }

    public ListF<Resource> findActiveResources(ListF<ResourceType> types) {
        return findResources(ResourceFields.IS_ACTIVE.eq(true).and(ResourceFields.TYPE.column().inSet(types)));
    }

    @RunWithRandomTest
    public ListF<Resource> findResourcesByDomain(PassportDomain domain) {
        return findResources(ResourceFields.DOMAIN.eq(domain.getDomain().getDomain()));
    }

    private Resource findResource(SqlCondition c) {
        ListF<Resource> found = findResources(c);

        if (found.isEmpty()) throw new EmptyResultDataAccessException(1);
        if (found.size() > 1) throw new IncorrectResultSizeDataAccessException(1);

        return found.single();
    }

    private ListF<Resource> findResourcesSorted(SqlCondition c, SqlOrder o) {
        return genericBeanDao.loadBeans(ResourceHelper.INSTANCE, c, o);
    }

    public ListF<Resource> findResources(SqlCondition c) {
        return genericBeanDao.loadBeans(ResourceHelper.INSTANCE, c);
    }

    public long saveResource(Resource template) {
        return genericBeanDao.insertBeanGetGeneratedKey(template);
    }

    @RunWithRandomTest
    public void updateResourcesSetInactive(SqlCondition condition) {
        Resource template = new Resource();
        template.setIsActive(false);
        genericBeanDao.updateBeans(template, condition);
    }

    public void updateResource(Resource resource) {
        genericBeanDao.updateBean(resource);
    }

    public int updateResourceByExchangeNameAndDomain(Resource resource, String exchangeName, PassportDomain passportDomain) {
        return genericBeanDao.updateBeans(resource,
                ResourceFields.EXCHANGE_NAME.eq(exchangeName).and(ResourceFields.DOMAIN.eq(passportDomain.getDomain().getDomain())));
    }

    @RunWithRandomTest
    public void updateOfficesSetInactive(SqlCondition condition) {
        Office template = new Office();
        template.setIsActive(false);
        genericBeanDao.updateBeans(template, condition);
    }

    public void updateOffices(ListF<Office> datas) {
        genericBeanDao.updateBeans(datas);
    }

    public int updateOfficeByInviteId(Office officeData, long inviteId) {
        return genericBeanDao.updateBeans(officeData, OfficeFields.INVITE_ID.eq(inviteId));
    }

    public int updateOfficeByCenterId(Office officeData, int centerId) {
        return genericBeanDao.updateBeans(officeData, OfficeFields.CENTER_ID.eq(centerId));
    }

    @RunWithRandomTest
    public Option<Office> findOfficeByInviteId(long inviteId) {
        String q = "SELECT * FROM office WHERE invite_id = ?";
        return getJdbcTemplate().queryForOption(q, Office.class, inviteId);
    }

    @RunWithRandomTest
    public Option<Office> findOfficeByCenterId(int centerId) {
        String q = "SELECT * FROM office WHERE center_id = ?";
        return getJdbcTemplate().queryForOption(q, Office.class, centerId);
    }

    @RunWithRandomTest
    public Option<Office> findOfficeById(long id) {
        return findOfficesByIds(Cf.list(id)).firstO();
    }

    @RunWithRandomTest
    public ListF<Office> findOfficesByIds(ListF<Long> ids) {
        String q = "SELECT * FROM office WHERE id " + SqlQueryUtils.inSet(ids);

        if (skipQuery(ids, q)) return Cf.list();

        return getJdbcTemplate().queryForList(q, Office.class);
    }

} //~
