package ru.yandex.calendar.logic.resource;

import java.util.regex.Pattern;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2B;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.regex.Matcher2;
import ru.yandex.misc.regex.Pattern2;

/**
* @author gutman
*/
public class ResourceFilter {

    private static final MapF<Character, Character> KEYBOARD_LAYOUT_REVERSION =
            Cf.charList("`qwertyuiop[]asdfghjkl;'zxcvbnm,.йцукенгшщзфывапролдячсмить".toCharArray())
                    .zip(Cf.charList("ёйцукенгшщзхъфывапролджэячсмитьбюqwertyuiopasdfghjklzxcvbnm".toCharArray()))
                    .toMap();

    private final ListF<ResourceCondition> resourceConditions;
    private final Option<String> resourceNamePrefix;
    private final Option<ListF<Email>> resourceEmailFilter;

    private ResourceFilter(ListF<ResourceCondition> resourceConditions,
            Option<String> resourceNamePrefix, Option<ListF<Email>> resourceEmailFilter)
    {
        if (resourceNamePrefix.isPresent()) {
            Validate.notEmpty(resourceNamePrefix.get());
        }
        this.resourceConditions = resourceConditions;
        this.resourceNamePrefix = resourceNamePrefix;
        this.resourceEmailFilter = resourceEmailFilter;
    }

    public static ResourceFilter any() {
        return new ResourceFilter(Cf.<ResourceCondition>list(), Option.<String>empty(), Option.<ListF<Email>>empty());
    }

    public static ResourceFilter byEmail(ListF<Email> emails) {
        return ResourceFilter.any().withEmailFilter(emails);
    }

    public ResourceFilter withFilter(String filter) {
        Validate.notEmpty(filter);
        return new ResourceFilter(parseFilter(filter), resourceNamePrefix, resourceEmailFilter);
    }

    public ResourceFilter withFilterSafe(String filter) {
        return filter.isEmpty() ? this : withFilter(filter);
    }

    public ResourceFilter withPrefix(String prefix) {
        Validate.notEmpty(prefix);
        return new ResourceFilter(resourceConditions, Option.of(prefix), resourceEmailFilter);
    }

    public ResourceFilter withEmailFilter(ListF<Email> emails) {
        return new ResourceFilter(resourceConditions, resourceNamePrefix, Option.of(emails));
    }

    public boolean hasVideoFilter() {
        return resourceConditions.containsTs(ResourceFieldCondition.HAS_VIDEO);
    }

    public Option<String> getResourceNamePrefix() {
        return resourceNamePrefix;
    }

    private static ListF<ResourceCondition> parseFilter(String filter) {
        ListF<String> titles = Cf.x(filter.split(",")).stableUnique();

        Tuple2<ResourceCapacityCondition, ListF<String>> t = ResourceCapacityCondition.consumeTitles(titles);
        ListF<ResourceCondition> fields = t._2.map(ResourceFieldCondition.fromTitleF()).uncheckedCast();
        ResourceCondition capacity = t._1;

        return fields.plus1(capacity);
    }

    public SqlCondition getSqlCondition() {
        return SqlCondition.all(resourceConditions.map(new Function<ResourceCondition, SqlCondition>() {
            public SqlCondition apply(ResourceCondition c) {
                return c.getSqlCondition();
            }
        }));
    }

    public Function1B<Resource> nameWordMatchesPrefixF() {
        if (!resourceNamePrefix.isPresent()) return Function1B.falseF();

        Pattern2 nextWordStartPattern = new Pattern2(Pattern.compile("[^\\p{L}]\\p{L}"));

        Function2B<String, String> matchesPrefixF = (name, prefix) -> {
            if (StringUtils.startsWithIgnoreCase(name, prefix)) {
                return true;
            }
            Matcher2 matcher = nextWordStartPattern.matcher2(name);
            while (matcher.find()) {
                if (StringUtils.startsWithIgnoreCase(name.substring(matcher.start() + 1), prefix)) {
                    return true;
                }
            }
            return false;
        };
        char[] namePrefixReChars = resourceNamePrefix.get().toCharArray();
        for (int i = 0; i < namePrefixReChars.length; ++i) {
            namePrefixReChars[i] = KEYBOARD_LAYOUT_REVERSION.getOrElse(namePrefixReChars[i], namePrefixReChars[i]);
        }
        Function1B<String> matchesF = matchesPrefixF.bind2(resourceNamePrefix.get())
                .orF(matchesPrefixF.bind2(new String(namePrefixReChars)));

        Function1B<Option<String>> matchesOF = o -> o.exists(matchesF);

        return Resource.getNameF().andThen(matchesOF).orF(Resource.getNameEnF().andThen(matchesOF));
    }

    public Function1B<ResourceInfo> getResourceNameFilter() {
        if (!resourceNamePrefix.isPresent()) {
            return Function1B.trueF();
        } else {
            String prefix = resourceNamePrefix.get();

            Function1B<Option<String>> startsWithF = o -> StringUtils.startsWithIgnoreCase(o.getOrElse(""), prefix);
            Function<String, String> f = s -> s.replaceAll("[\\p{L}\\s]", "");
            Function1B<Option<String>> notAlphaOrSpaceStartsWithF = startsWithF.compose(a -> a.map(f));

            return ResourceInfo.resourceF().andThen(Function1B.anyOfF(
                    nameWordMatchesPrefixF(),
                    Resource.getAlterNameF().andThen(notAlphaOrSpaceStartsWithF),
                    Resource.getExchangeNameF().andThen(startsWithF)));
        }
    }

    public Option<ListF<Email>> getResourceEmailFilter() {
        return resourceEmailFilter;
    }

    public boolean sameAs(ResourceFilter that) {
        return this.resourceNamePrefix.equals(that.resourceNamePrefix)
                && this.resourceConditions.unique().equals(that.resourceConditions.unique());
    }

    private static enum ResourceFieldCondition implements ResourceCondition{
        HAS_VIDEO(ResourceFields.VIDEO.gt(0), "video"),
        HAS_DESK(ResourceFields.DESK.eq(true), "desk"),
        HAS_GUEST_WIFI(ResourceFields.GUEST_WIFI.eq(true), "guest_wifi"),
        HAS_PROJECTOR(ResourceFields.PROJECTOR.gt(0), "projector"),
        HAS_LCD_PANEL(ResourceFields.LCD_PANEL.gt(0), "lcd_panel"),
        HAS_PHONE(ResourceFields.PHONE.column().isNotNull(), "phone"),
        HAS_VOICE_CONFERENCING(ResourceFields.VOICE_CONFERENCING.eq(true), "voice_conferencing"),
        HAS_MARKER_BOARD(ResourceFields.MARKER_BOARD.eq(true), "marker_board"),
        MULTI_OFFICE(SqlCondition.trueCondition(), "multi-office"),
        ;

        private final SqlCondition sqlCondition;
        private final String title;

        private ResourceFieldCondition(SqlCondition sqlCondition, String title) {
            this.sqlCondition = sqlCondition;
            this.title = title;
        }

        @Override
        public SqlCondition getSqlCondition() {
            return sqlCondition;
        }

        public static Function<ResourceFieldCondition, String> getTitleF() {
            return new Function<ResourceFieldCondition, String>() {
                public String apply(ResourceFieldCondition resourceCondition) {
                    return resourceCondition.title;
                }
            };
        }

        private static final MapF<String, ResourceFieldCondition> byTitle = Cf.x(values()).toMapMappingToKey(getTitleF());

        public static Function<String, ResourceFieldCondition> fromTitleF() {
            return t -> byTitle.getOrThrow(t, "Unexpected resource field: ", t);
        }
    }

    private static class ResourceCapacityCondition extends DefaultObject implements ResourceCondition {
        private static final MapF<String, SqlCondition> byTitle = Cf.toMap(Tuple2List.fromPairs(
                "large", ResourceFields.CAPACITY.ge(12),
                "medium", ResourceFields.CAPACITY.ge(10).and(ResourceFields.CAPACITY.lt(12)),
                "small", ResourceFields.CAPACITY.lt(10)));

        private final SetF<String> titles;

        private ResourceCapacityCondition(ListF<String> titles) {
            this.titles = titles.unique();
        }

        @Override
        public SqlCondition getSqlCondition() {
            return titles.isEmpty()
                    ? SqlCondition.trueCondition()
                    : SqlCondition.any(titles.map(t -> byTitle.getOrThrow(t, "Unexpected resource size: ", t)));
        }

        public static Tuple2<ResourceCapacityCondition, ListF<String>> consumeTitles(ListF<String> titles) {
            Tuple2<ListF<String>, ListF<String>> t = titles.partition(byTitle::containsKeyTs);
            return Tuple2.tuple(new ResourceCapacityCondition(t._1), t._2);
        }
    }

    private interface ResourceCondition {
        SqlCondition getSqlCondition();
    }

    public static Function1B<ResourceFilter> hasVideoFilterF() {
        return new Function1B<ResourceFilter>() {
            public boolean apply(ResourceFilter f) {
                return f.hasVideoFilter();
            }
        };
    }
}
