package ru.yandex.calendar.logic.resource;

import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.a3.error.ReadableErrorMessageSource;
import ru.yandex.calendar.frontend.bender.UnicodeEmailMarshaller;
import ru.yandex.calendar.frontend.bender.UnicodeEmailUnmarshaller;
import ru.yandex.calendar.frontend.bender.WebDateTime;
import ru.yandex.calendar.frontend.bender.WebDateTimeMarshallerUnmarshaller;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.util.dates.DateInterval;
import ru.yandex.calendar.util.dates.DateOrDateTime;
import ru.yandex.calendar.util.dates.TimeField;
import ru.yandex.calendar.util.dates.TimesInUnit;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderIgnore;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.email.Email;

/**
 * @author dbrylev
 */
@BenderBindAllFields
public class ResourceInaccessibility implements ReadableErrorMessageSource {

    public static final BenderParserSerializer<ResourceInaccessibility> ps = Bender.cons(
            ResourceInaccessibility.class, BenderConfiguration.cons(
                    MembersToBind.WITH_ANNOTATIONS, false,
                    CustomMarshallerUnmarshallerFactoryBuilder.cons()
                            .add(Email.class, new UnicodeEmailMarshaller(), new UnicodeEmailUnmarshaller())
                            .add(WebDateTime.class, new WebDateTimeMarshallerUnmarshaller(), new WebDateTimeMarshallerUnmarshaller())
                            .build()));

    @BenderIgnore
    private final Resource resource;

    private final long id;
    private final Email email;

    private final Reason reason;
    private final Option<String> message;
    private final Option<Integer> maxLengthMinutes;
    private final Option<Integer> minLengthMinutes;
    private final Option<WebDateTime> maxStart;
    private final Option<DateInterval> restrictionDates;
    private final Option<TimesInUnit> maxTimesInUnit;

    private ResourceInaccessibility(
            Resource resource, Reason reason,
            Option<String> message, Option<Duration> maxLength, Option<Duration> minLength, Option<DateTime> maxStart,
            Option<DateInterval> restrictionDates, Option<TimesInUnit> maxTimesInUnit) {
        this.resource = resource;
        this.id = resource.getId();
        this.email = ResourceRoutines.getResourceEmail(resource);
        this.reason = reason;
        this.message = message;
        this.maxLengthMinutes = maxLength.map(Duration::getStandardMinutes).map(Cf.Long.toIntegerF());
        this.minLengthMinutes = minLength.map(Duration::getStandardMinutes).map(Cf.Long.toIntegerF());
        this.maxStart = maxStart.map(WebDateTime::dateTime);
        this.restrictionDates = restrictionDates;
        this.maxTimesInUnit = maxTimesInUnit;
    }

    public static ResourceInaccessibility repetitionDenied(Resource resource) {
        return new ResourceInaccessibility(resource, Reason.REPETITION_DENIED,
                Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.empty());
    }

    public static ResourceInaccessibility tooLongEvent(Resource resource, Duration limit) {
        return new ResourceInaccessibility(resource, Reason.TOO_LONG_EVENT,
                Option.empty(), Option.of(limit), Option.empty(), Option.empty(), Option.empty(), Option.empty());
    }

    public static ResourceInaccessibility tooShortEvent(Resource resource, Duration limit) {
        return new ResourceInaccessibility(resource, Reason.TOO_SHORT_EVENT,
                Option.empty(), Option.empty(), Option.of(limit), Option.empty(), Option.empty(), Option.empty());
    }

    public static ResourceInaccessibility tooFarEvent(Resource resource, DateTime maxStart) {
        return new ResourceInaccessibility(resource, Reason.TOO_FAR_EVENT,
                Option.empty(), Option.empty(), Option.empty(), Option.of(maxStart), Option.empty(), Option.empty());
    }

    public static ResourceInaccessibility dateRestricted(Resource resource, DateInterval dates) {
        return new ResourceInaccessibility(resource, Reason.DATE_RESTRICTED,
                Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.of(dates), Option.empty());
    }

    public static ResourceInaccessibility zeroLengthEvent(Resource resource) {
        return new ResourceInaccessibility(resource, Reason.ZERO_LENGTH_EVENT,
                Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.empty());
    }

    public static ResourceInaccessibility tooManyEvents(Resource resource, TimesInUnit timesInUnit) {
        return new ResourceInaccessibility(resource, Reason.TOO_MANY_EVENTS,
                Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.of(timesInUnit));
    }

    public static ResourceInaccessibility tooManyEventsByUser(Resource resource, TimesInUnit timesInUnit) {
        return new ResourceInaccessibility(resource, Reason.TOO_MANY_EVENTS_BY_USER,
                Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.of(timesInUnit));
    }

    public static ResourceInaccessibility massageDenied(Resource resource, String error) {
        return new ResourceInaccessibility(resource, Reason.MASSAGE_DENIED,
                Option.of(error), Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.empty());
    }

    public Resource getResource() {
        return resource;
    }

    public long getResourceId() {
        return id;
    }

    public Email getEmail() {
        return email;
    }

    public Reason getReason() {
        return reason;
    }

    public Option<String> getMessage() {
        return message;
    }

    public Option<Integer> getMaxLengthMinutes() {
        return maxLengthMinutes;
    }

    public Option<Integer> getMinLengthMinutes() {
        return minLengthMinutes;
    }

    public Option<Instant> getMaxStartTs() {
        return maxStart.map(dt -> dt.getDateTime().toInstant());
    }

    public Option<DateInterval> getRestrictionDates() {
        return restrictionDates;
    }

    public Option<TimesInUnit> getMaxTimesInUnit() {
        return maxTimesInUnit;
    }

    @Override
    public NameI18n getReadableMessage() {
        NameI18n name = resource.getName().map(n -> new NameI18n(n, resource.getNameEn()))
                .getOrElse(NameI18n.empty());

        switch (reason) {
            case REPETITION_DENIED:
                return new NameI18n(
                        "Бронирование повторяющихся событий в переговорке «$1» невозможно",
                        "Impossible to create repeating event at room «$1»")
                        .replace("$1", name);
            case TOO_LONG_EVENT:
                return new NameI18n(
                        "Длительность события в переговорке «$1» не должна превышать",
                        "The event length at room «$1» cannot exceed")
                        .concat(durationInMinutesToReadableString(getMaxLengthMinutes().get()))
                        .replace("$1", name)
                        .replace("$2", NameI18n.constant(getMaxLengthMinutes().get().toString()));
            case TOO_SHORT_EVENT:
                return new NameI18n(
                        "Длительность события в переговорке «$1» должна быть больше чем",
                        "The event length at room «$1» must be more than")
                        .concat(durationInMinutesToReadableString(getMinLengthMinutes().get()))
                        .replace("$1", name)
                        .replace("$2", NameI18n.constant(getMinLengthMinutes().get().toString()));
            case TOO_FAR_EVENT:
                return new NameI18n("Переговорка «$1» бронируема до $2", "Room «$1» can be booked up to $2")
                        .replace("$1", name)
                        .replace("$2", NameI18n.constant(maxStart.get().toLocalDateTime().toString("yyyy-MM-dd HH:mm")));

            case DATE_RESTRICTED:
                DateInterval dates = getRestrictionDates().get();

                Option<LocalDate> startDate = dates.start.map(d -> d.toLocalDateTime().toLocalDate());
                Option<LocalDateTime> start = dates.start.map(DateOrDateTime::toLocalDateTime);

                Option<LocalDate> endDate = dates.end.map(d -> d.toLocalDateTime().toLocalDate());
                Option<LocalDateTime> end = dates.end.map(DateOrDateTime::toLocalDateTime);

                NameI18n range;

                if (startDate.exists(endDate::isSome)) {
                    range = NameI18n.constant(" " + startDate.get().toString()
                            + " " + start.get().toString("HH:mm") + "–" + end.get().toString("HH:mm"));

                } else {
                    Option<NameI18n> st = start.map(d -> new NameI18n(" с ", " since ")
                            .concat(NameI18n.constant(d.toString("yyyy-MM-dd HH:mm"))));

                    Option<NameI18n> en = end.map(d -> new NameI18n(" до ", " till ")
                            .concat(NameI18n.constant(d.toString("yyyy-MM-dd HH:mm"))));

                    range = st.plus(en).reduceLeftO(NameI18n::concat).getOrElse(NameI18n.empty());
                }
                if (dates.title.isPresent()) {
                    return dates.title.get().concat(new NameI18n(" в переговорке «$1»", " at room «$1»"))
                            .concat(new NameI18n(", недоступна", ", unbookable"))
                            .concat(range).replace("$1", name);
                } else {
                    return new NameI18n("Переговорка «$1» недоступна", "Room «$1» is unbookable")
                            .concat(range).replace("$1", name);
                }
            case ZERO_LENGTH_EVENT:
                return new NameI18n(
                        "Длительность события слишком мала, выберите другое время",
                        "Period of time is too short. Please choose another one");

            case TOO_MANY_EVENTS:
            case TOO_MANY_EVENTS_BY_USER:
                int times = getMaxTimesInUnit().get().getTimes();
                TimeField field = getMaxTimesInUnit().get().getUnit();

                NameI18n unit =
                        field == TimeField.DAY ? new NameI18n(" в день", " per day")
                                : field == TimeField.WEEKDAYS ? new NameI18n(" в будние дни", " per weekdays")
                                : field == TimeField.WEEK ? new NameI18n(" в неделю", " per week")
                                : field == TimeField.MONTH ? new NameI18n(" в месяц", " per month")
                                : new NameI18n("", "");

                NameI18n limit = times == 1
                        ? new NameI18n(" больше 1 раза", " more than 1 time")
                        : new NameI18n(" больше " + times + " раз", " more than " + times + " times");

                if (reason == Reason.TOO_MANY_EVENTS) {
                    return new NameI18n("Переговорка «$1» может быть забронирована не", "Room «$1» cannot be booked")
                            .concat(limit).concat(unit).replace("$1", name);
                } else {
                    return new NameI18n("Вы можете бронировать переговорку «$1» не", "You cannot book «$1»")
                            .concat(limit).concat(unit).replace("$1", name);
                }

            case MASSAGE_DENIED:
                return new NameI18n(
                        "Бронирование нескольких сеансов массажа на будущее осуществляется"
                                + " сотрудниками отдела компенсаций и льгот при наличии направления на массажный курс.",
                        "Reservations for several massage sessions are made by employees of the compensations"
                                + " and benefits department if the massage course is available.");
        }
        throw new IllegalStateException("Unexpected reason " + reason);
    }

    private NameI18n durationInMinutesToReadableString(int length) {
        Option<NameI18n> days = Option.of(length / 1440).filter(i -> i > 0)
                .map(NameI18n.plural("день", "дня", "дней", "day", "days"))
                .map(NameI18n.constant(" ")::concat);

        Option<NameI18n> hours = Option.of(length % 1440 / 60).filter(i -> i > 0)
                .map(NameI18n.plural("час", "часа", "часов", "hour", "hours"))
                .map(NameI18n.constant(" ")::concat);

        Option<NameI18n> minutes = Option.of(length % 60).filter(i -> i > 0)
                .map(NameI18n.plural("минуту", "минуты", "минут", "minute", "minutes"))
                .map(NameI18n.constant(" ")::concat);

        return days.plus(hours).plus(minutes).reduceLeft(NameI18n::concat);
    }

    public enum Reason {
        REPETITION_DENIED,
        TOO_LONG_EVENT,
        TOO_SHORT_EVENT,
        TOO_FAR_EVENT,
        DATE_RESTRICTED,
        ZERO_LENGTH_EVENT,
        TOO_MANY_EVENTS,
        TOO_MANY_EVENTS_BY_USER,
        MASSAGE_DENIED,
    }
}
