package ru.yandex.calendar.logic.resource;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.logic.beans.generated.Office;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.inside.passport.blackbox.PassportDomain;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.ip.InternetDomainName;
import ru.yandex.misc.lang.DefaultToString;

/**
 * @author Stepan Koltsov
 */
public class ResourceInfo extends DefaultToString {
    private final Resource resource;
    private final Office office;

    public ResourceInfo(Resource resource, Office office) {
        this.resource = resource;
        this.office = office;
    }

    public Resource getResource() {
        return resource;
    }

    public Office getOffice() {
        return office;
    }

    public Option<String> getName() {
        return resource.getName();
    }

    public Option<String> getNameWithAlterName() {
        return nameWithAlterNameF().apply(this);
    }

    public Option<String> getPhone() {
        return resource.getPhone().map(Cf.Integer.toStringF());
    }

    public Option<String> getVideo() {
        return resource.getVideo().map(Cf.Integer.toStringF());
    }

    public Option<String> getNameI18n(Language lang) {
        return ResourceRoutines.getNameI18n(resource, lang);
    }

    public Option<String> getAlterNameI18n(Language lang) {
        return ResourceRoutines.getAlterNameI18n(resource, lang);
    }

    public String getOfficeNameI18n(Language lang) {
        return ResourceRoutines.getNameI18n(office, lang);
    }

    public Option<String> getCityNameI18n(Language lang) {
        return ResourceRoutines.getCityNameI18n(office, lang);
    }

    public long getResourceId() {
        return resource.getId();
    }

    public long getOfficeId() {
        return resource.getOfficeId();
    }

    public Option<Long> getOfficeStaffId() {
        return getOffice().getStaffId();
    }

    public Option<Integer> getFloorNum() {
        return resource.getFloorNum();
    }

    public Option<String> getGroupNameI18n(Language lang) {
        return ResourceRoutines.getGroupNameI18n(resource, lang);
    }

    public Option<Integer> getCapacity() {
        return resource.getCapacity();
    }

    public String getExchangeName() {
        return getEmail().getLocalPart();
    }

    public Email getEmail() {
        return ResourceRoutines.getResourceEmail(resource);
    }

    public Option<NameI18n> getNameWithAlterNameI18n() {
        return ResourceRoutines.getNameWithAlterNameI18n(getResource());
    }

    public Option<String> getStaffMapLink() {
        if (resource.getMapUrl().isPresent()) {
            return resource.getMapUrl();
        }
        InternetDomainName domain = new InternetDomainName(getResource().getDomain());

        if (this.isMeetingRoom()
            && resource.getExistsOnStaff()
            && PassportAuthDomain.byDomain(domain) == PassportAuthDomain.YANDEX_TEAM_RU)
        {
            return getResource().getExchangeName()
                    .map(s -> "https://staff.yandex-team.ru/map/#/conference_room/name_exchange/" + s);
        }
        return Option.empty();
    }

    public boolean isMeetingRoom() {
        return resource.getType() == ResourceType.ROOM
                || resource.getType() == ResourceType.PROTECTED_ROOM
                || resource.getType() == ResourceType.PRIVATE_ROOM
                || resource.getType() == ResourceType.YAMONEY_ROOM;
    }

    public static Function1B<ResourceInfo> isYaTeamF() {
        return new Function1B<ResourceInfo>() {
            public boolean apply(ResourceInfo a) {
                final PassportDomain domain = PassportDomain.cons(a.getResource().getDomain());
                return PassportDomain.YANDEX_TEAM_RU.equals(domain);
            }
        };
    }

    public static Function1B<ResourceInfo> isSyncWithExchangeF() {
        return new Function1B<ResourceInfo>() {
            public boolean apply(ResourceInfo resourceInfo) {
                return resourceInfo.getResource().getSyncWithExchange();
            }
        };
    }

    public static Function1B<ResourceInfo> isAsyncWithExchangeF() {
        return new Function1B<ResourceInfo>() {
            public boolean apply(ResourceInfo r) {
                return r.getResource().getAsyncWithExchange();
            }
        };
    }

    public static Function<ResourceInfo, String> nameF() {
        return new Function<ResourceInfo, String>() {
            public String apply(ResourceInfo a) {
                return a.getName().getOrNull();
            }
        };
    }

    public static Function<ResourceInfo, Option<String>> getNameWithAlterNameI18nF(final Language lang) {
        return new Function<ResourceInfo, Option<String>>() {
            public Option<String> apply(ResourceInfo r) {
                return r.getNameWithAlterNameI18n().map(NameI18n.getNameF(lang));
            }
        };
    }

    public static Function<ResourceInfo, Option<String>> getStaffMapLinkF() {
        return new Function<ResourceInfo, Option<String>>() {
            public Option<String> apply(ResourceInfo r) {
                return r.getStaffMapLink();
            }
        };
    }

    public static Function<ResourceInfo, Option<String>> nameWithAlterNameF() {
        return new Function<ResourceInfo, Option<String>>() {
            public Option<String> apply(ResourceInfo a) {
                return a.getNameWithAlterNameI18n().map(NameI18n.getNameF(Language.RUSSIAN));
            }
        };
    }

    public static Function<ResourceInfo, Resource> resourceF() {
        return new Function<ResourceInfo, Resource>() {
            public Resource apply(ResourceInfo a) {
                return a.resource;
            }
        };
    }

    public static Function<ResourceInfo, Long> resourceIdF() {
        return resourceF().andThen(ResourceFields.ID.getF());
    }

    public static Function<ResourceInfo, Office> officeF() {
        return new Function<ResourceInfo, Office>() {
            public Office apply(ResourceInfo a) {
                return a.office;
            }
        };
    }

    public static Function<ResourceInfo, Long> officeIdF() {
        return new Function<ResourceInfo, Long>() {
            public Long apply(ResourceInfo resource) {
                return resource.getOfficeId();
            }
        };
    }

    public static Function1B<ResourceInfo> officeIdIsF(long officeId) {
        return officeIdF().andThenEquals(officeId);
    }

    public static Function<ResourceInfo, Option<String>> officeCityNameF() {
        return new Function<ResourceInfo, Option<String>>() {
            public Option<String> apply(ResourceInfo r) {
                return r.getOffice().getCityName();
            }
        };
    }

    public static Function1B<ResourceInfo> officeCityNameIsF(Option<String> cityName) {
        return officeCityNameF().andThenEquals(cityName);
    }

    public static Function<ResourceInfo, Option<Integer>> floorNumF() {
        return new Function<ResourceInfo, Option<Integer>>() {
            public Option<Integer> apply(ResourceInfo resource) {
                return resource.getFloorNum();
            }
        };
    }

    public static Function<ResourceInfo, Option<Integer>> getCapacityF() {
        return new Function<ResourceInfo, Option<Integer>>() {
            public Option<Integer> apply(ResourceInfo resource) {
                return resource.getCapacity();
            }
        };
    }

    public static Function1B<ResourceInfo> hasVideoF() {
        return r -> r.getResource().getVideo().isPresent();
    }

    public static Function<ResourceInfo, Email> emailF() {
        return new Function<ResourceInfo, Email>() {
            public Email apply(ResourceInfo resource) {
                return resource.getEmail();
            }
        };
    }

    public static Function<ResourceInfo, Option<String>> exchangeNameF() {
        return resourceF().andThen(Resource.getExchangeNameF());
    }

    public static Function1B<ResourceInfo> isActiveF() {
        return Function1B.wrap(resourceF().andThen(Resource.getIsActiveF()));
    }

    public static Function2<Resource, Office, ResourceInfo> consF() {
        return new Function2<Resource, Office, ResourceInfo>() {
            public ResourceInfo apply(Resource resource, Office office) {
                return new ResourceInfo(resource, office);
            }
        };
    }

    public static Function1B<ResourceInfo> isYaTeamAndSyncWithExchange() {
        return isYaTeamF().andF(isSyncWithExchangeF());
    }

    public static Function<ResourceInfo, Option<String>> getNameI18nF(final Language lang) {
        return new Function<ResourceInfo, Option<String>>() {
            public Option<String> apply(ResourceInfo r) {
                return r.getNameI18n(lang);
            }
        };
    }

    public static Function1B<ResourceInfo> isMeetingRoomF() {
        return new Function1B<ResourceInfo>() {
            public boolean apply(ResourceInfo r) {
                return r.isMeetingRoom();
            }
        };
    }

    public static Function1B<ResourceInfo> isTypeF(final ResourceType resourceType) {
        return new Function1B<ResourceInfo>() {
            public boolean apply(ResourceInfo r) {
                return r.getResource().getType() == resourceType;
            }
        };
    }

} //~
