package ru.yandex.calendar.logic.resource;

import javax.annotation.Nullable;

import lombok.val;
import org.joda.time.DateTimeZone;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2B;
import ru.yandex.calendar.logic.beans.generated.Office;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.contact.Contact;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.participant.ParticipantsData;
import ru.yandex.calendar.logic.sharing.participant.ResourceParticipantInfo;
import ru.yandex.calendar.logic.update.LockResource;
import ru.yandex.calendar.logic.update.PgTransactionLocker;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.SpecialUsers;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.wiki.WikiApiClient;
import ru.yandex.calendar.util.wiki.WikiUtils;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.dynproperties.DynamicPropertyRegistry;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.inside.passport.blackbox.PassportDomain;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.ip.InternetDomainName;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

public class ResourceRoutines {
    public static final InternetDomainName YT_DOMAIN = PassportAuthDomain.YANDEX_TEAM_RU.getDomain();
    public static final InternetDomainName YT_DEV_DOMAIN = new InternetDomainName("msft.yandex-team.ru");
    private static final String RESOURCE_PREFIX = "calendar-resource-"; // avoid coincidence

    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private OfficeManager officeManager;
    @Autowired
    private EnvironmentType environmentType;
    @Autowired
    private UserManager userManager;
    @Autowired
    private WikiApiClient wikiApiClient;
    @Autowired
    private DynamicPropertyRegistry dynamicPropertyRegistry;
    @Autowired
    private PgTransactionLocker pgTransactionLocker;

    private final DynamicProperty<ListF<Long>> autoDeclineFromDisplayResourceIds =
            new DynamicProperty<>("displayAutoDeclineResourceIds", Cf.list());

    public static PassportUid getMasterOfResources(PassportDomain domain) {
        if (domain.sameAs(PassportDomain.YANDEX_TEAM_RU)) {
            return SpecialUsers.MASTER_OF_RESOURCES_YT;
        } else {
            throw new IllegalArgumentException();
        }
    }

    /**
     * @param uid specifies user's domain
     * @return domain string for domains that support resources
     */
    public Option<PassportDomain> getDomainByUidUnlessPublic(PassportUid uid) {
        switch (uid.getGroup()) {
        case DOMAINS: {
            val domain = settingsRoutines.getSettingsByUid(uid).getCommon().getDomain().getOrNull();
            Validate.notEmpty(domain);
            return Option.of(PassportDomain.cons(domain));
        }
        case YANDEX_TEAM_RU: {
            return Option.of(PassportDomain.YANDEX_TEAM_RU);
        }
        default: {
            return Option.empty();
        }
        } // switch
    }

    public PassportUid getUidOrMasterOfResource(UidOrResourceId uidOrResourceId, EventData eventData) {
        if (uidOrResourceId.isResource()) {
            return eventData.getInvData().getParticipantsDataO()
                    .filterMap(ParticipantsData::getOrganizerEmailSafe).filterMap(userManager::getUidByEmail)
                    .getOrElse(() -> {
                        Resource resource = resourceDao.findResourceById(uidOrResourceId.getResourceId());
                        return ResourceRoutines.getMasterOfResources(PassportDomain.cons(resource.getDomain()));
                    });
        } else {
            return uidOrResourceId.getUid();
        }
    }

    public Resource loadById(long id) {
        return resourceDao.findResourceById(id);
    }

    public ListF<ResourceInfo> getDomainResourcesCanBookWithLayersAndOffices(
            final PassportUid uid, Tuple2List<Long, ResourceFilter> officeFilters)
    {
        return getDomainResourcesWithLayersAndOffices(uid, officeFilters, true);
    }

    public ListF<ResourceInfo> getDomainResourcesCanViewWithLayersAndOffices(
            final PassportUid uid, Tuple2List<Long, ResourceFilter> officeFilters)
    {
        return getDomainResourcesWithLayersAndOffices(uid, officeFilters, false);
    }

    public ListF<ResourceInfo> getDomainResourcesCanBookWithLayersAndOffices(
            PassportUid uid, OfficeFilter officeFilter, ResourceFilter resourceFilter)
    {
        return getDomainResourcesWithLayersAndOffices(uid, officeFilter, resourceFilter, true);
    }

    public ListF<ResourceInfo> getDomainResourcesCanViewWithLayersAndOffices(
            PassportUid uid, OfficeFilter officeFilter, ResourceFilter resourceFilter)
    {
        return getDomainResourcesWithLayersAndOffices(uid, officeFilter, resourceFilter, false);
    }

    private ListF<ResourceInfo> getDomainResourcesWithLayersAndOffices(
            final PassportUid uid, Tuple2List<Long, ResourceFilter> officeFilters, final boolean isForBook)
    {
        return officeFilters.flatMap(of -> getDomainResourcesWithLayersAndOffices(uid, OfficeFilter.byId(of._1), of._2, isForBook));
    }

    private ListF<ResourceInfo> getDomainResourcesWithLayersAndOffices(
            PassportUid uid, OfficeFilter officeFilter, ResourceFilter resourceFilter, boolean isForBook)
    {
        val domainO = getDomainByUidUnlessPublic(uid);

        if (!domainO.isPresent()) return Cf.list();
        val domain = domainO.get();

        SqlCondition resourceCondition = resourceFilter.getSqlCondition();
        if (resourceFilter.getResourceEmailFilter().isPresent()) {
            val localParts = resourceFilter.getResourceEmailFilter().get()
                    .filterMap(e -> Option.when(e.getDomain().equals(domain.getDomain()), e.getLocalPart()));
            resourceCondition = resourceCondition.and(ResourceFields.EXCHANGE_NAME.column().inSet(localParts));
        }
        val user = userManager.getUserInfo(uid);

        Function<Function2B<ResourceType, Option<Integer>>, Function1B<ResourceInfo>> permittedF =
                f -> resource -> f.apply(resource.getResource().getType(), resource.getResource().getAccessGroup());

        resourceCondition = resourceCondition.and(ResourceFields.IS_ACTIVE.eq(true));
        resourceCondition = resourceCondition.and(ResourceFields.TYPE.column()
                .inSet(isForBook ? user.getResourceTypesCanBook() : user.getResourceTypesCanView()));

        val officeCondition = officeFilter.getSqlCondition();

        return resourceDao
                .findDomainResourcesWithLayersAndOffices(domain, officeCondition, resourceCondition)
                .filter(resourceFilter.getResourceNameFilter()
                        .andF(permittedF.apply(isForBook ? user::canBookResource : user::canViewResource)));
    }

    public ListF<ResourceInfo> getYtRoomsHaveDisplayToken() {
        return getYtRooms(ResourceFields.DISPLAY_TOKEN.column().isNotNull());
    }

    public ListF<ResourceInfo> getYtActiveRoomsWithoutDisplayToken() {
        return getYtRooms(ResourceFields.DISPLAY_TOKEN.column().isNull()
                .and(ResourceFields.IS_ACTIVE.eq(true)
                        .or(ResourceFields.EXCHANGE_NAME.eq(SpecialResources.DISPLAY_ROOM_EMAIL.getLocalPart()))));
    }

    private ListF<ResourceInfo> getYtRooms(SqlCondition condition) {
        val resourceCondition = ResourceFields.TYPE.column()
                .inSet(Cf.list(ResourceType.ROOM, ResourceType.YAMONEY_ROOM,
                        ResourceType.PROTECTED_ROOM, ResourceType.PRIVATE_ROOM)).and(condition);

        return resourceDao.findDomainResourcesWithLayersAndOffices(
                PassportDomain.YANDEX_TEAM_RU, SqlCondition.trueCondition(), resourceCondition);
    }

    public Option<ResourceInfo> findResourceInfoByDisplayToken(String token) {
        val resource = resourceDao.findResourceByDisplayToken(token);
        val office = resourceDao.findOfficesByIds(resource.map(Resource.getOfficeIdF())).singleO();

        return resource.zip(office).map(ResourceInfo.consF()).singleO();
    }

    public void updateResourceDisplayTokenById(long id, @Nullable String token) {
        val data = new Resource();
        data.setId(id);
        data.setDisplayToken(token);

        resourceDao.updateResource(data);
    }

    public void updateResource(Resource data) {
        resourceDao.updateResource(data);
    }

    public ListF<Resource> getResourcesByIds(ListF<Long> resourceIds) {
        val idsUnique = resourceIds.unique();
        val found = resourceDao.findResourcesByIds(resourceIds);

        Validate.sameSize(idsUnique, found, "resources not found by some of ids " + idsUnique);
        return found;
    }

    public ListF<ResourceInfo> getResourceInfosByIds(ListF<Long> resourceIds) {
        return resourceDao.findResourceInfosByIds(resourceIds);
    }

    public ListF<Resource> findActiveRooms() {
        return findActiveResources(Cf.list(ResourceType.ROOM));
    }

    public ListF<Resource> findActiveResources(ListF<ResourceType> types) {
        return resourceDao.findActiveResources(types);
    }

    public Option<Resource> findResourceByDomainAndExchangeName(InternetDomainName domain, String name) {
        return resourceDao.findResourceByDomainAndExchangeName(domain, name);
    }

    public DateTimeZone getTimeZoneByResourceId(long resourceId) {
        return getTimeZonesByResourceIds(Cf.list(resourceId)).single().get2();
    }

    public Tuple2List<Long, DateTimeZone> getTimeZonesByResourceIds(ListF<Long> resourceIds) {
        return resourceDao.findResourceInfosByIds(resourceIds).toTuple2List(
                ResourceInfo.resourceIdF(),
                ResourceInfo.officeF().andThen(OfficeManager.getOfficeTimeZoneF()));
    }

    public ListF<Contact> getDomainResourcesAsContacts(PassportUid uid) {
        return getDomainResourcesCanBookWithLayersAndOffices(uid, OfficeFilter.any(), ResourceFilter.any()).map(
                r -> new Contact(
                        getResourceEmail(r.getResource()),
                        r.getName().getOrElse("?") // resource alter name or ...
                    ));
    }

    /**
     * creates resource email (real for YT with exchange name; fake otherwise)
     */
    public static Email getResourceEmail(Resource resource) {
        val localPart = hasExchangeName(resource) ?
                resource.getExchangeName().getOrNull() : RESOURCE_PREFIX + resource.getId();
        return new Email(localPart + "@" + resource.getDomain());
    }

    public static Function<Resource, Email> getResourceEmailF() {
        return ResourceRoutines::getResourceEmail;
    }

    public Email getResourceEmailByResourceId(long resourceId) {
        return getResourceEmail(loadById(resourceId));
    }

    public Email getExchangeEmailById(long resourceId) {
        return getExchangeEmail(loadById(resourceId));
    }

    public Email getExchangeEmail(Resource resource) {
        Validate.isTrue(hasExchangeName(resource));
        val domain = Cf.list(EnvironmentType.DEVELOPMENT, EnvironmentType.TESTS, EnvironmentType.TESTING)
                .containsTs(environmentType) ?
                "msft.yandex-team.ru" : resource.getDomain();
        return new Email(resource.getExchangeName().getOrNull() + "@" + domain);

    }

    public Tuple2List<Long, Option<ResourceType>> findResourceTypesByIds(ListF<Long> ids) {
        val found = resourceDao.findResourcesByIds(ids);
        return ids.zipWith(found.toMap(Resource.getIdF(), Resource.getTypeF())::getO);
    }

    public Tuple2List<Email, Option<Resource>> findResourcesByEmails(ListF<Email> emails) {
        val idByEmail = Cf2.flatBy2(parseIdsFromEmails(emails)).toMap();

        val resourceById = resourceDao.findResourcesByIds(idByEmail.values().toList())
                .toMapMappingToKey(Resource.getIdF());

        return emails.zipWith(e -> idByEmail.getO(e).map(resourceById::getOrThrow));
    }

    public ListF<Email> selectResourceEmails(ListF<Email> emails) {
        return Cf2.flatBy2(parseIdsFromEmails(emails)).get1();
    }

    public ListF<Email> selectNotActiveResourceEmails(ListF<Email> emails) {
        return Cf2.flatBy2(findResourcesByEmails(emails))
                .filterBy2(Function1B.wrap(Resource.getIsActiveF()).notF()).get1();
    }

    public Option<Long> parseIdFromEmail(Email email) {
        return parseIdsFromEmails(Cf.list(email)).get2().single();
    }

    public Tuple2List<Email, Option<Long>> parseIdsFromEmails(ListF<Email> emails) {
        MapF<Email, Long> idByEmail = Cf.hashMap();

        val grouped = emails.groupBy(Email::getDomain);
        for (Tuple2<InternetDomainName, ListF<Email>> t : grouped.entries()) {
            // we could also verify domain, but, in fact, there is no need to do that (ssytnik@)
            val domain = t._1;
            val localParts = t._2.map(Email::getLocalPart);

            val p = localParts.partition(s -> s.startsWith(RESOURCE_PREFIX));

            final var parsed = p._1
                    .zipWithFlatMapO(s -> Cf.Long.parseSafe(StringUtils.substring(s, RESOURCE_PREFIX.length())));

            val fixedDomain = isYaTeamDevelopmentDomain(domain) ? YT_DOMAIN : domain;
            val found = resourceDao.findResourcesByDomainAndExchangeNames(fixedDomain, p._2)
                    .map2(Resource::getId);

            idByEmail.putAll(parsed.plus(found).map1(s -> new Email(s + "@" + domain)));
        }
        return emails.zipWith(idByEmail::getO);
    }

    public boolean isResource(Email guestEmail) {
        // XXX: evil
        val resourceIdO = parseIdFromEmail(guestEmail);
        return resourceIdO.isPresent();
    }

    public void lockResourcesByIds(ListF<Long> ids) {
        pgTransactionLocker.lock(ids.map(id -> new LockResource(LockResource.Type.RESOURCE, id)));
    }

    public ListF<Long> tryLockResourcesByIds(ListF<Long> ids) {
        return ids.zip(pgTransactionLocker.tryLock(ids.map(id -> new LockResource(LockResource.Type.RESOURCE, id))))
                .filterMap(t -> Option.when(t.get2(), t.get1()));
    }

    public Function1B<Email> isResourceF() {
        return this::isResource;
    }

    public Option<Long> getResourceId(Email email) {
        val localPart = email.getLocalPart();
        val domain = email.getDomain();
        if (domain.sameAs(YT_DOMAIN) || isYaTeamDevelopmentDomain(domain)) {
            return findByExchangeName(localPart)
                    .map(ResourceFields.ID.getF());
        } else {
            return resourceDao.findResourceByDomainAndExchangeName(email.getDomain(), email.getLocalPart())
                    .map(ResourceFields.ID.getF());
        }
    }

    private boolean isYaTeamDevelopmentDomain(final InternetDomainName domain) {
        // hack: test exchange use "@msft.yandex-team.ru" domain for resources
        return Cf.list(EnvironmentType.DEVELOPMENT, EnvironmentType.TESTS).containsTs(environmentType) && domain.sameAs(YT_DEV_DOMAIN);
    }

    public static boolean isYaTeam(Resource resource) {
        return resource.getDomain().equalsIgnoreCase(YT_DOMAIN.getDomain());
    }

    public static boolean isYaTeamDev(Resource resource) {
        return resource.getDomain().equalsIgnoreCase(YT_DEV_DOMAIN.getDomain());
    }

    public static boolean hasExchangeName(Resource resource) {
        return StringUtils.isNotEmpty(resource.getFieldValueO(ResourceFields.EXCHANGE_NAME).getOrElse(""));
    }

    public boolean hasExchangeName(long resourceId) {
        return hasExchangeName(loadById(resourceId));
    }

    // TODO create refreshable {exchangeName - resource} cache to avoid db access,
    // this would greatly optimize repeatable email-is-a-resource check
    public Option<Resource> findByExchangeName(String exchangeName) {
        return resourceDao.findResourceByDomainAndExchangeName(YT_DOMAIN, exchangeName);
    }

    public ListF<Resource> findByExchangeName(ListF<String> exchangeNames) {
        return resourceDao.findResourcesByDomainAndExchangeNames(YT_DOMAIN, exchangeNames).get2();
    }

    public Function<String, Option<Resource>> findByExchangeNameF() {
        return this::findByExchangeName;
    }

    public Option<Resource> findByExchangeEmail(Email email) {
        return findByExchangeName(email.getLocalPart());
    }

    public boolean existsByExchangeEmail(Email email) {
        return findByExchangeEmail(email).isPresent();
    }

    public long createResource(Resource template) {
        return resourceDao.saveResource(template);
    }

    public Option<String> locationStringWithResourceNames(PassportUid uid, Participants participants) {
        val resourceInfos = participants
                .getParticipantsSafe()
                .filterByType(ResourceParticipantInfo.class)
                .map(ResourceParticipantInfo.resourceInfoF())
                ;
        if (resourceInfos.isNotEmpty()) {
            return Option.of(sortResourcesFromUserOfficeAndCityFirst(uid, resourceInfos)
                    .filterMap(ResourceInfo.nameWithAlterNameF()).mkString(", "));
        } else {
            return Option.empty();
        }
    }

    public Option<Office> getUserOffice(PassportUid uid) {
        val sts = settingsRoutines.getSettingsByUidIfExists(uid);
        val userOfficeId = sts.filterMap(SettingsInfo.getYtF()).filterMap(SettingsYt.getActiveOfficeIdF());

        return officeManager.getOfficesByIds(userOfficeId).singleO();
    }

    public ListF<ResourceInfo> sortResourcesFromUserOfficeAndCityFirst(PassportUid uid, ListF<ResourceInfo> resources) {
        val office = getUserOffice(uid);
        if (office.isPresent()) {
            val hasIdF = ResourceInfo.officeIdIsF(office.get().getId());
            val hasCityNameF = ResourceInfo.officeCityNameIsF(office.get().getCityName());

            return resources.filter(hasIdF)
                    .plus(resources.filter(hasIdF.notF().andF(hasCityNameF)))
                    .plus(resources.filter(hasIdF.orF(hasCityNameF).notF()));
        } else {
            return resources;
        }
    }

    public ListF<ResourceInfo> selectResourcesFromUserOfficeOrCityOrGetAll(PassportUid uid, ListF<ResourceInfo> resources) {
        val office = getUserOffice(uid);

        if (!office.isPresent()) return resources;

        ListF<ResourceInfo> found = resources.filter(ResourceInfo.officeIdIsF(office.get().getId()));

        if (found.isNotEmpty()) return found;

        found = resources.filter(ResourceInfo.officeCityNameIsF(office.get().getCityName()));

        if (found.isNotEmpty()) return found;

        return resources;
    }

    public static Option<NameI18n> getNameWithAlterNameI18n(Resource resource) {
        val name = resource.getName();
        val nameEn = resource.getNameEn();
        val alterName = resource.getAlterName();
        val alterNameEn = resource.getAlterNameEn();

        if (!name.isPresent()) return Option.empty();

        val suffix = alterName.isPresent() ? alterName.mkString(" (", "", ")") : "";
        val suffixEn = alterNameEn.isPresent() ? alterNameEn.mkString(" (", "", ")") : suffix;

        return Option.of(new NameI18n(name.get() + suffix, nameEn.map(s -> s + suffixEn)));
    }

    public static Option<String> getNameI18n(Resource resource, Language lang) {
        return nameI18n(resource.getName(), resource.getNameEn()).map(NameI18n.getNameF(lang));
    }

    public static Option<String> getAlterNameI18n(Resource resource, Language lang) {
        return nameI18n(resource.getAlterName(), resource.getAlterNameEn()).map(NameI18n.getNameF(lang));
    }

    public static Option<String> getCityNameI18n(Office office, Language lang) {
        return nameI18n(office.getCityName(), office.getCityNameEn()).map(NameI18n.getNameF(lang));
    }

    public static String getNameI18n(Office office, Language lang) {
        return new NameI18n(office.getName(), Option.of(office.getNameEn())).getName(lang);
    }

    public static Option<String> getGroupNameI18n(Resource resource, Language lang) {
        return nameI18n(resource.getGroupName(), resource.getGroupNameEn()).map(NameI18n.getNameF(lang));
    }

    public static Option<String> getProtectionMessageI18n(Resource resource, Language lang) {
        return nameI18n(resource.getProtectionMessage(), resource.getProtectionMessageEn()).map(NameI18n.getNameF(lang));
    }

    private static Option<NameI18n> nameI18n(Option<String> name, Option<String> enName) {
        return name.isPresent() ? Option.of(new NameI18n(name.get(), enName)) : Option.empty();
    }

    public Function<Long, ResourceInfo> getResourceInfoByIdF() {
        return resourceId -> resourceDao.findResourceInfoByResourceId(resourceId);
    }

    public String getApartmentRulesHtml(String apartmentResourceExchangeName) {
        if (environmentType == EnvironmentType.TESTS) {
            return "";
        }
        try {
            return WikiUtils.stripHideRefererFromLinks(WikiUtils.cleanHeads(
                    wikiApiClient.getFormatted("/Corpapartments/ApartmentRule/" + apartmentResourceExchangeName)));
        } catch (Exception e) {
            throw new RuntimeException(
                    "Error while getting apartment rules for exchange name=" + apartmentResourceExchangeName, e);
        }
    }

    public boolean isAutoDeclineFromDisplay(long resourceId) {
        return getAutoDeclineFromDisplayResourceIds().containsTs(resourceId);
    }

    public ListF<Long> getAutoDeclineFromDisplayResourceIds() {
        return autoDeclineFromDisplayResourceIds.get();
    }

    public synchronized void updateResourceAutoDeclineFromDisplay(long resourceId, boolean decline) {
        val resourceIds = Cf.toArrayList(autoDeclineFromDisplayResourceIds.get());

        if (decline == resourceIds.containsTs(resourceId)) return;

        if (decline) {
            resourceIds.add(resourceId);
        } else {
            resourceIds.removeTs(resourceId);
        }
        dynamicPropertyRegistry.setValue(autoDeclineFromDisplayResourceIds, resourceIds);
    }

    public Function<String, String> getApartmentRulesHtmlF() {
        return this::getApartmentRulesHtml;
    }
}
