package ru.yandex.calendar.logic.resource;

import java.util.Collection;
import java.util.function.Function;

import lombok.Data;
import lombok.Value;
import lombok.val;
import one.util.streamex.StreamEx;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.util.dates.DateInterval;
import ru.yandex.calendar.util.dates.TimesInUnit;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.annotation.BenderTextValue;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.regex.AsteriskPattern;
import ru.yandex.misc.time.InstantInterval;

public class SpecialResources {
    public static final Email DISPLAY_ROOM_EMAIL = new Email("conf_st_display@yandex-team.ru");

    public static final DynamicProperty<ListF<String>> noSuggestInRooms =
            new DynamicProperty<>("rooms.noSuggest", Cf.list(
                    "conference_asm", // 3.Мазуров
                    "conf_rr_1_4", // 1.Класс
                    "conf_rr_1_6", // 1.Экстрополис
                    "conf_rr_1_8", // Массаж
                    "conf_rr_3_14", // 3.Врач
                    "conf_rr_5_33", // Ещё один массаж
                    "conf_rr_7_6", // 7.Небо
                    "conf_st_5_3", // Дзэн
                    "conf_st_muz", // Музыкальная комната
                    "conf_st_yoga", // Комната йоги
                    "conf_spb_massage", // Массажный Кабинет
                    "dancing-hall", // Зал для танцев
                    "conf_ekb_greenwich", // Библиотека и она же музыкальная комната
                    "conf_ekb_sport", // Спортзал
                    "conf_sim_library", // Библиотека
                    "conf_sim_gym", // Спортзал
                    "conf_videostudio"
            ));

    public static final DynamicProperty<ListF<String>> repetitionUnacceptableRooms =
            new DynamicProperty<>("rooms.repetitionUnacceptable", Cf.list(
                    "conf_st_muz", // Музыкальная комната
                    "conf_rr_1_29", // Мулен Руж
                    "conf_rr_1_8", "conf_rr_5_33", "conf_spb_massage" // Массажные кабинеты
            ));

    public static final DynamicProperty<ListF<String>> guestsDescriptionRequiredRooms =
            new DynamicProperty<>("rooms.guestsDescriptionRequired", Cf.list());

    public static final DynamicProperty<ListF<RoomsDuration>> durationLimitedRooms =
            new DynamicProperty<>(
                "rooms.durationLimited",
                Cf.list(),
                RoomsDuration::isValidList
            );

    public static final DynamicProperty<ListF<RoomsDuration>> minDurationLimitedRooms =
            new DynamicProperty<>(
                    "rooms.minDurationLimited",
                    Cf.list(),
                    RoomsDuration::isValidList
            );

    public static final DynamicProperty<ListF<RoomsDuration>> eventDistanceLimitedRooms =
            new DynamicProperty<>(
                "rooms.eventDistanceLimited",
                Cf.list(),
                RoomsDuration::isValidList
            );

    public static final DynamicProperty<ListF<RoomsDates>> dateRestrictedRooms =
            new DynamicProperty<>("rooms.dateRestricted", Cf.list());

    public static final DynamicProperty<ListF<RoomsEvents>> eventsLimitedRooms =
            new DynamicProperty<>("rooms.eventsLimited", Cf.list());

    public static final DynamicProperty<ListF<RoomsEvents>> eventsByUserLimitedRooms =
            new DynamicProperty<>("rooms.eventsByUserLimited", Cf.list());

    public static final DynamicProperty<ListF<RoomsDisplaySettings>> displaySettings =
            new DynamicProperty<>("rooms.displaySettings", Cf.list(new RoomsDisplaySettings(
                    new Rooms(Cf.list(), Cf.list(), Cf.list(), Cf.list()), 10, 15, 3)));

    public static boolean isRepetitionUnacceptable(Resource resource) {
        return repetitionUnacceptableRooms.get().containsTs(resource.getExchangeName().getOrElse(""));
    }

    public static Option<Duration> getDurationLimit(UserInfo user, Resource resource) {
        if (user.canIgnoreDurationLimit(resource)) {
            return Option.empty();
        }

        return getDurationLimit(resource);
    }

    public static Option<Duration> getDurationLimit(Resource resource) {
        return findDuration(resource, durationLimitedRooms.get());
    }

    public static Option<Duration> getMinimalDurationLimit(UserInfo user, Resource resource) {
        if (user.canIgnoreDurationLimit(resource)) {
            return Option.empty();
        }

        return getMinimalDurationLimit(resource);
    }

    public static Option<Duration> getMinimalDurationLimit(Resource resource) {
        return findDuration(resource, minDurationLimitedRooms.get());
    }

    public static Option<DateTime> getEventMaxStart(UserInfo user, ResourceInfo resource, Instant now) {
        if (user.canIgnoreMaxEventStart(resource.getResource())) {
            return Option.empty();
        }
        return getEventMaxStart(resource, now);
    }

    public static Option<DateTime> getEventMaxStart(ResourceInfo resource, Instant now) {
        DateTimeZone tz = OfficeManager.getOfficeTimeZone(resource.getOffice());

        return findDuration(resource.getResource(), eventDistanceLimitedRooms.get())
                .map(maxDistance -> now.toDateTime(tz).plus(maxDistance).minuteOfHour().roundFloorCopy());
    }

    public static ListF<InstantInterval> getRestrictionIntervals(ResourceInfo resource) {
        DateTimeZone tz = OfficeManager.getOfficeTimeZone(resource.getOffice());

        return getRestrictionDates(resource).map(i -> i.toInstantInterval(tz));
    }

    public static ListF<DateInterval> getRestrictionDates(ResourceInfo resource) {
        return findDates(resource.getResource(), dateRestrictedRooms.get());
    }

    static Option<Duration> findDuration(Resource resource, Collection<RoomsDuration> rooms) {
        return Option.x(
                rooms.stream()
                    .filter(rs -> rs.rooms.matches(resource))
                    .map(RoomsDuration::getDuration)
                    .findFirst()
        );
    }

    private static ListF<DateInterval> findDates(Resource resource, ListF<RoomsDates> rooms) {
        return rooms.filter(rs -> rs.rooms.matches(resource)).map(RoomsDates::getDates);
    }

    public static ListF<TimesInUnit> getEventsLimits(Resource resource) {
        return findEventsLimits(resource, eventsLimitedRooms.get());
    }

    public static ListF<TimesInUnit> getEventsLimitsPerUser(Resource resource) {
        return findEventsLimits(resource, eventsByUserLimitedRooms.get());
    }

    public static Option<InstantInterval> findMaxEventsLimitCheckInterval(Resource resource, DateTime dateTime) {
        return getEventsLimits(resource).plus(SpecialResources.getEventsLimitsPerUser(resource))
                .filterMap(o -> o.getUnit().expand(dateTime))
                .maxByO(InstantInterval::getDuration);
    }

    public static RoomsDisplaySettings getDefaultDisplaySettings() {
        return displaySettings.get().find(rs -> rs.rooms.isEmpty()).getOrThrow("No default display settings found");
    }

    public static RoomsDisplaySettings getDisplaySettings(Resource resource) {
        return displaySettings.get().find(rs -> !rs.rooms.isEmpty() && rs.rooms.matches(resource))
                .getOrElse(SpecialResources::getDefaultDisplaySettings);
    }

    private static ListF<TimesInUnit> findEventsLimits(Resource resource, ListF<RoomsEvents> rooms) {
        return rooms.filter(r -> r.rooms.matches(resource)).map(RoomsEvents::getEvents);
    }

    @BenderBindAllFields
    @Value
    public static class RoomsDuration {
        @BenderFlatten
        Rooms rooms;
        int minutes;

        public RoomsDuration(Rooms rooms, Duration limit) {
            this(rooms, (int) limit.getStandardMinutes());
        }

        public RoomsDuration(Rooms rooms, int minutes) {
            this.rooms = rooms;
            this.minutes = minutes;
        }

        public Rooms getRooms() {
            return rooms;
        }

        public int getMinutes() {
            return minutes;
        }

        public Duration getDuration() {
            return Duration.standardMinutes(minutes);
        }

        public boolean isValid() {
            return !this.rooms.isEmpty() && this.minutes > 0;
        }

        public boolean isNotValid() {
            return !isValid();
        }

        static boolean isValidList(Collection<RoomsDuration> roomsDurations) {
            val nonUniqueRooms = StreamEx.of(roomsDurations)
                .flatMap(roomsDuration -> roomsDuration.rooms.rooms.stream())
                .flatCollection(Function.identity())
                .distinct(2);

            return StreamEx.of(roomsDurations).allMatch(RoomsDuration::isValid) && nonUniqueRooms.count() == 0;
        }
    }

    @BenderBindAllFields
    public static class RoomsDates {
        @BenderFlatten
        private final Rooms rooms;
        @BenderFlatten
        private final DateInterval dates;

        public RoomsDates(Rooms rooms, DateInterval dates) {
            this.rooms = rooms;
            this.dates = dates;
        }

        public Rooms getRooms() {
            return rooms;
        }

        public DateInterval getDates() {
            return dates;
        }
    }

    @BenderBindAllFields
    public static class RoomsEvents {
        @BenderFlatten
        private final Rooms rooms;
        @BenderFlatten
        private final TimesInUnit events;

        public RoomsEvents(Rooms rooms, TimesInUnit events) {
            this.rooms = rooms;
            this.events = events;
        }

        public Rooms getRooms() {
            return rooms;
        }

        public TimesInUnit getEvents() {
            return events;
        }
    }

    @Data
    @BenderBindAllFields
    public static class RoomsDisplaySettings {
        @BenderFlatten
        private final Rooms rooms;

        private final int checkInStart;
        private final int checkInEnd;
        private final int timesToTotalDecline;
    }

    @BenderBindAllFields
    @Value
    public static class Rooms {
        @BenderPart(wrapperName = "rooms")
        private final Option<ListF<RoomPattern>> rooms;
        @BenderPart(wrapperName = "accessGroups")
        private final Option<ListF<Integer>> accessGroups;
        @BenderPart(wrapperName = "resourceTypes")
        private final Option<ListF<ResourceType>> resourceTypes;
        @BenderPart(wrapperName = "offices")
        private final Option<ListF<Long>> offices;

        public Rooms(ListF<String> rooms, ListF<Integer> accessGroups, ListF<ResourceType> resourceTypes, ListF<Long> resourceIds) {
            this.rooms = Option.when(rooms.isNotEmpty(), rooms.map(RoomPattern::new));
            this.accessGroups = Option.when(accessGroups.isNotEmpty(), accessGroups);
            this.resourceTypes = Option.when(resourceTypes.isNotEmpty(), resourceTypes);
            this.offices = Option.when(resourceTypes.isNotEmpty(), resourceIds);
        }

        public static Rooms of(Resource resource) {
            return new Rooms(
                    resource.getFieldValueO(ResourceFields.EXCHANGE_NAME),
                    resource.getFieldValueO(ResourceFields.ACCESS_GROUP),
                    resource.getFieldValueO(ResourceFields.TYPE),
                    resource.getFieldValueO(ResourceFields.OFFICE_ID)
            );
        }

        public boolean isEmpty() {
            return !rooms.isPresent() && !accessGroups.isPresent() && !resourceTypes.isPresent() && !offices.isPresent();
        }

        public boolean matches(Resource resource) {
            return !offices.isMatch(rs -> !rs.containsTs(resource.getOfficeId()))
                    && !rooms.isMatch(rs -> !rs.exists(r -> resource.getExchangeName().isMatch(r::matches))
                    && !accessGroups.isMatch(gs -> !gs.exists(resource.getAccessGroup()::isSome))
                    && !resourceTypes.isMatch(ts -> !ts.containsTs(resource.getType())));
        }
    }

    @Value
    public static class RoomPattern {
        AsteriskPattern pattern;

        private RoomPattern(String value) {
            this.pattern = new AsteriskPattern(value);
        }

        public boolean matches(String name) {
            return pattern.matches(name);
        }

        @BenderTextValue
        public static RoomPattern parse(String value) {
            return new RoomPattern(value);
        }

        @BenderTextValue
        public String serialize() {
            return pattern.getPattern();
        }
    }
}
