package ru.yandex.calendar.logic.resource;

import lombok.EqualsAndHashCode;

import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.ResourceParticipantInfo;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.Validate;

/**
 * Either user, or resource.
 * @see ParticipantId
 */
@EqualsAndHashCode(callSuper = true)
public class UidOrResourceId extends DefaultObject {
    private final Either<PassportUid, Long> uidOrResourceId;

    private UidOrResourceId(Either<PassportUid, Long> either) {
        this.uidOrResourceId = either;
    }

    public static UidOrResourceId fromOptions(Option<PassportUid> uid, Option<Long> resourceId) {
        return new UidOrResourceId(Either.fromOptions(uid, resourceId));
    }

    public static UidOrResourceId user(PassportUid uid) {
        return new UidOrResourceId(Either.left(uid));
    }

    public static Function<PassportUid, UidOrResourceId> userF() {
        return UidOrResourceId::user;
    }

    public static UidOrResourceId resource(ResourceParticipantInfo resource) {
        return resource(resource.getResourceId());
    }

    public static UidOrResourceId resource(long id) {
        Validate.isTrue(id > 0);
        return new UidOrResourceId(Either.right(id));
    }

    public static Function<Long, UidOrResourceId> resourceF() {
        return UidOrResourceId::resource;
    }

    /** some if uid is defined, none otherwise. Applicable unless user can be anonymous. */
    public static Option<UidOrResourceId> notNullUserO(Option<PassportUid> uidO) {
        return uidO.map(userF());
    }

    public boolean isUser() {
        return uidOrResourceId.isLeft();
    }

    public boolean isResource() {
        return uidOrResourceId.isRight();
    }

    public static Function1B<UidOrResourceId> isUserF() {
        return UidOrResourceId::isUser;
    }

    public PassportUid getUid() {
        return uidOrResourceId.getLeft();
    }

    public Option<PassportUid> getUidO() {
        return uidOrResourceId.leftO();
    }

    public long getResourceId() {
        return uidOrResourceId.getRight();
    }

    public Option<Long> getResourceIdO() {
        return uidOrResourceId.rightO();
    }

    public static Function<UidOrResourceId, PassportUid> getUidF() {
        return UidOrResourceId::getUid;
    }

    public static Function<UidOrResourceId, Option<PassportUid>> getUidOF() {
        return UidOrResourceId::getUidO;
    }

    public static Function<UidOrResourceId, Long> getResourceIdF() {
        return UidOrResourceId::getResourceId;
    }

    public static Function<UidOrResourceId, Option<Long>> getResourceIdOF() {
        return UidOrResourceId::getResourceIdO;
    }

    public ParticipantId toParticipantId() {
        if (isUser()) {
            return ParticipantId.yandexUid(getUid());
        } else if (isResource()) {
            return ParticipantId.resourceId(getResourceId());
        } else {
            throw new IllegalStateException();
        }
    }

    @Override
    public String toString() {
        return isUser() ? "User(" + getUid() + ")" : "Resource(" + getResourceId() + ")";
    }
}
