package ru.yandex.calendar.logic.resource.center;

import java.util.List;
import java.util.Optional;
import java.util.Set;

import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.ews.subscriber.ExchangeSubscriber;
import ru.yandex.calendar.logic.beans.generated.Office;
import ru.yandex.calendar.logic.beans.generated.OfficeFields;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.resource.ResourceType;
import ru.yandex.calendar.logic.staff.dao.OfficesDao;
import ru.yandex.calendar.logic.staff.dao.RoomsDao;
import ru.yandex.calendar.micro.yt.entity.YtOffice;
import ru.yandex.calendar.micro.yt.entity.YtRoom;
import ru.yandex.inside.passport.blackbox.PassportDomain;
import ru.yandex.mail.cerberus.LocationId;
import ru.yandex.misc.TranslitUtils;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.StringUtils;

import static java.util.function.Predicate.not;

@Slf4j
public class CenterResourceUpdater {
    @Autowired
    private ExchangeSubscriber exchangeSubscriber;
    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    OfficesDao officesDao;
    @Autowired
    RoomsDao roomsDao;


    private static final Set<String> COUNTRIES = Set.of(
            "Украина", "Беларусь", "США", "Турция", "Швейцария",
            "Ukraine", "Belarus", "USA", "Turkey", "Switzerland");

    private static StreamEx<String> split(String name) {
        return StreamEx.of(name.split(", "))
                .dropWhile(COUNTRIES::contains);
    }

    @SneakyThrows
    private Office convert(YtOffice ytOffice) {
        val info = ytOffice.getInfo();

        val office = new Office();
        office.setAbbr(info.getCode());
        office.setCenterId((int) ytOffice.getId().getValue());
        office.setStaffId(ytOffice.getId().getValue());

        split(info.getName().getRu())
                .peekFirst(office::setCityName)
                .peekLast(office::setName)
                .forEach(ignored -> {
                });

        split(info.getName().getEn())
                .peekFirst(office::setCityNameEn)
                .peekLast(office::setNameEn)
                .forEach(ignored -> {
                });

        office.setTimezoneId(info.getTimezone().getId());
        return office;
    }

    @SneakyThrows
    private List<Office> fetchOfficesByIds(Set<LocationId> officeIds) {

        return StreamEx.of(officesDao.getAll())
                .filter(office -> officeIds.contains(office.getId()))
                .map(this::convert)
                .toImmutableList();
    }

    @SneakyThrows
    private Resource convert(YtRoom room) {
        val info = room.getInfo();

        val resource = new Resource();
        resource.setStaffId(room.getId().getValue());
        resource.setFloorId(info.getFloorId());
        resource.setFloorNum(Option.x(info.getFloorNumber()));

        resource.setName(Option.x(Optional.of(info.getRuName()).filter(not(String::isEmpty))));
        resource.setNameEn(Option.x(Optional.of(info.getEnName()).filter(not(String::isEmpty))));
        resource.setAlterName(StringUtils.notBlankO(info.getAlternativeName()));

        resource.setCapacity(Option.when(info.getCapacity() > 0, info.getCapacity()));
        resource.setDescription(info.getAdditional() == null ? "" : info.getAdditional());

        resource.setExchangeName(room.getName().toLowerCase());
        resource.setDesk(info.getEquipment().isDesk());
        resource.setLcdPanel(info.getEquipment().getScreen());
        resource.setProjector(info.getEquipment().getProjector());
        resource.setMarkerBoard(info.getEquipment().isMarkerBoard());

        resource.setPhone(Cf.Integer.parseSafe(info.getPhone()));
        resource.setSeats(Option.when(info.getEquipment().getSeats() > 0, info.getEquipment().getSeats()));
        resource.setVideo(Cf.Integer.parseSafe(info.getEquipment().getVideoConferencing()));
        resource.setVoiceConferencing(info.getEquipment().isVoiceConferencing());
        resource.setGuestWifi(info.getEquipment().isGuestWifi());

        resource.setIsActive(info.isBookable() && room.isActive());
        return resource;
    }

    @SneakyThrows
    public void update() {
        Check.isTrue(passportAuthDomainsHolder.containsYandexTeamRu(), "No Center in public");

        val rooms = roomsDao.getAll();
        val officeIds = StreamEx.of(rooms)
                .flatMap(room -> room.getLocationId().stream())
                .toImmutableSet();

        val offices = fetchOfficesByIds(officeIds);
        updateOffices(offices);

        val officeIdByStaffId = StreamEx.of(offices)
                .mapToEntry(Office::getCenterId, Office::getId)
                .flatMapKeys(CollectionF::stream)
                .mapKeys(Long::valueOf)
                .toImmutableMap();

        val resources = StreamEx.of(rooms)
                .mapToEntry(YtRoom::getLocationId, this::convert)
                .mapKeyValue((staffOfficeIdOpt, resource) -> {
                    staffOfficeIdOpt
                            .map(LocationId::getValue)
                            .map(officeIdByStaffId::get)
                            .ifPresent(resource::setOfficeId);
                    return resource;
                })
                .toImmutableList();

        updateResources(resources);
        exchangeSubscriber.updateAllResourceSubscriptions();
    }

    private void updateOffices(List<Office> offices) {
        for (Office office : offices) {
            office.setDomain(PassportDomain.YANDEX_TEAM_RU.getDomain().getDomain());

            int updated = resourceDao.updateOfficeByCenterId(office, office.getCenterId().get());

            if (updated == 0) {
                resourceDao.saveOffice(office);
            }
            office.setId(resourceDao.findOfficeByCenterId(office.getCenterId().get()).get().getId());
        }
        SqlCondition c = OfficeFields.ID.column().inSet(StreamEx.of(offices).map(Office::getId).toImmutableSet()).not();
        resourceDao.updateOfficesSetInactive(c);
    }

    private void updateResources(List<Resource> resources) {
        for (Resource resource : resources) {
            resource.setDomain(PassportDomain.YANDEX_TEAM_RU.getDomain().getDomain());

            int updated = resourceDao.updateResourceByExchangeNameAndDomain(
                    resource, resource.getExchangeName().get(), PassportDomain.YANDEX_TEAM_RU);

            if (updated == 0) {
                if (resource.getAlterName().isPresent()) {
                    resource.setFieldValueDefault(ResourceFields.ALTER_NAME_EN,
                            TranslitUtils.translit(resource.getAlterName().get()));
                }
                resourceDao.saveResource(resource);
            }
        }

        val set = StreamEx.of(resources)
                .flatMap(r -> r.getExchangeName().stream())
                .toImmutableSet();
        SqlCondition c = ResourceFields.DOMAIN.eq(PassportDomain.YANDEX_TEAM_RU.getDomain().getDomain())
                .and(ResourceFields.EXCHANGE_NAME.column().inSet(set).not())
                .and(ResourceFields.TYPE.column().inSet(Cf.list(
                        ResourceType.ROOM, ResourceType.MASSAGE_ROOM,
                        ResourceType.PROTECTED_ROOM, ResourceType.PRIVATE_ROOM, ResourceType.YAMONEY_ROOM)))
                .and(ResourceFields.EXISTS_ON_STAFF.eq(true));

        resourceDao.updateResourcesSetInactive(c);
    }

    void setMicroContextForTest(OfficesDao officesDao, RoomsDao roomsDao) {
        this.roomsDao = roomsDao;
        this.officesDao = officesDao;
    }
}
