package ru.yandex.calendar.logic.resource.reservation;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.GenericBeanDao;
import ru.yandex.calendar.logic.beans.generated.ResourceReservation;
import ru.yandex.calendar.logic.beans.generated.ResourceReservationFields;
import ru.yandex.calendar.logic.beans.generated.ResourceReservationHelper;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author dbrylev
 */
public class ResourceReservationDao extends CalendarJdbcDaoSupport {

    @Autowired
    private GenericBeanDao genericBeanDao;

    private SqlCondition withinInterval(InstantInterval interval) {
        SqlCondition singleReservationC = ResourceReservationFields.END_TS.gt(interval.getStart())
                .and(ResourceReservationFields.START_TS.lt(interval.getEnd()))
                .and(ResourceReservationFields.R_RULE.column().isNull());

        SqlCondition repeatingReservationC = ResourceReservationFields.START_TS.lt(interval.getEnd())
                .and(ResourceReservationFields.R_RULE.column().isNotNull())
                .and(ResourceReservationFields.R_DUE_TS.column().isNull()
                        .or(ResourceReservationFields.R_DUE_TS.gt(interval.getStart())));

        return singleReservationC.or(repeatingReservationC);
    }

    @RunWithRandomTest
    public ListF<ResourceReservation> findByResourceIdsAndIntervalAndDeadlinedAfterAndNeCreatorUid(
            ListF<Long> resourceIds, InstantInterval interval, Instant deadline, Option<PassportUid> exceptUid)
    {
        SqlCondition commonReservationC = ResourceReservationFields.RESOURCE_ID.column().inSet(resourceIds)
                .and(ResourceReservationFields.DEADLINE.ge(deadline))
                .and(Option.when(exceptUid.isPresent(), () -> ResourceReservationFields.CREATOR_UID.ne(exceptUid.get())));

        return genericBeanDao.loadBeans(ResourceReservationHelper.INSTANCE, commonReservationC
                .and(withinInterval(interval)));
    }

    @RunWithRandomTest
    public ListF<ResourceReservation> findByResourceIdsAndIntervalAndCreatorUid(
            ListF<Long> resourceIds, InstantInterval interval, PassportUid uid)
    {
        SqlCondition commonReservationC = ResourceReservationFields.RESOURCE_ID.column().inSet(resourceIds)
                .and(ResourceReservationFields.CREATOR_UID.eq(uid));

        return genericBeanDao.loadBeans(ResourceReservationHelper.INSTANCE, commonReservationC
                .and(withinInterval(interval)));
    }

    public ListF<ResourceReservation> findByIdAndCreatorUid(long id, PassportUid uid) {
        return genericBeanDao.loadBeans(ResourceReservationHelper.INSTANCE, idAndCreatorUidAre(id, uid));
    }

    public void insertBatch(ListF<ResourceReservation> reservations) {
        genericBeanDao.insertBeans(reservations);
    }

    public void updateByIdCreatorUidAndResourceId(ListF<ResourceReservation> reservations) {
        genericBeanDao.updateBeans(reservations);
    }

    public void deleteByIdAndCreatorUid(long id, PassportUid uid) {
        genericBeanDao.deleteBeans(ResourceReservationHelper.INSTANCE, idAndCreatorUidAre(id, uid));
    }

    public void deleteByIdCreatorUidAndResourceIds(long id, PassportUid uid, ListF<Long> resourceIds) {
        genericBeanDao.deleteBeans(
                ResourceReservationHelper.INSTANCE, idAndCreatorUidAre(id, uid).and(resourceIdIn(resourceIds)));
    }

    public void deleteByCreatorUid(PassportUid uid) {
        genericBeanDao.deleteBeans(ResourceReservationHelper.INSTANCE, ResourceReservationFields.CREATOR_UID.eq(uid));
    }

    public void deleteByResourceIds(ListF<Long> resourceIds) {
        genericBeanDao.deleteBeans(ResourceReservationHelper.INSTANCE, resourceIdIn(resourceIds));
    }

    public void deleteDeadlinedBefore(Instant now) {
        genericBeanDao.deleteBeans(ResourceReservationHelper.INSTANCE, ResourceReservationFields.DEADLINE.lt(now));
    }

    private SqlCondition idAndCreatorUidAre(long id, PassportUid uid) {
        return ResourceReservationFields.RESERVATION_ID.eq(id).and(ResourceReservationFields.CREATOR_UID.eq(uid));
    }

    private SqlCondition resourceIdIn(ListF<Long> resourceIds) {
        return ResourceReservationFields.RESOURCE_ID.column().inSet(resourceIds);
    }
}
