package ru.yandex.calendar.logic.resource.reservation;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.ResourceReservation;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.ics.iv5j.ical.type.recur.IcsRecur;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author dbrylev
 */
public class ResourceReservationManager {

    @Autowired
    private ResourceReservationDao resourceReservationDao;

    private final DynamicProperty<Integer> resourceReservationDurationSec =
            new DynamicProperty<>("resourceReservationDurationSec", 5 * 60);


    public ListF<ResourceReservationInfo> findReservations(
            ListF<Long> resourceIds, InstantInterval interval,
            Option<PassportUid> exceptUid, ActionInfo actionInfo)
    {
        ListF<ResourceReservation> reservations =
                resourceReservationDao.findByResourceIdsAndIntervalAndDeadlinedAfterAndNeCreatorUid(
                        resourceIds, interval, actionInfo.getNow(), exceptUid);

        return reservations.filterMap(r -> {
            ResourceReservationInfo info = new ResourceReservationInfo(r);
            ListF<InstantInterval> intervals = RepetitionUtils.getIntervals(
                    info.getRepetitionInfo(), interval.getStart(), Option.of(interval.getEnd()), true, 1);

            return Option.when(intervals.isNotEmpty(), info);
        });
    }

    public long createOrUpdateReservations(
            PassportUid creatorUid, long reservationId, ListF<Long> resourceIds,
            RepetitionInstanceInfo repetitionInfo, ActionInfo actionInfo)
    {

        Option<String> repetition = RepetitionUtils.getRecurWithExcludesAndNoDue(repetitionInfo).map(IcsRecur::getValue);
        Option<Instant> dueTs = repetitionInfo.getRepetition().filterMap(Repetition.getDueTsF());

        ListF<ResourceReservation> found = resourceReservationDao.findByIdAndCreatorUid(reservationId, creatorUid);

        Instant deadline = found.isEmpty()
                ? actionInfo.getNow().plus(Duration.standardSeconds(resourceReservationDurationSec.get()))
                : found.first().getDeadline();

        ListF<ResourceReservation> datas = resourceIds.stableUnique().map(rId -> {
            ResourceReservation data = new ResourceReservation();
            data.setReservationId(reservationId);
            data.setStartTs(repetitionInfo.getEventInterval().getStart());
            data.setEndTs(repetitionInfo.getEventInterval().getEnd());
            data.setResourceId(rId);
            data.setCreatorUid(creatorUid);
            data.setRRule(repetition);
            data.setRDueTs(dueTs);
            data.setTimezoneId(repetitionInfo.getTz().getID());
            data.setDeadline(deadline);

            return data;
        });
        Function<ResourceReservation, Long> resourceIdF = ResourceReservation.getResourceIdF();
        SetF<Long> foundResourceIds = found.map(resourceIdF).unique();

        ListF<ResourceReservation> creates = datas.filter(resourceIdF.andThen(foundResourceIds.containsF().notF()));
        ListF<ResourceReservation> updates = datas.filter(resourceIdF.andThen(foundResourceIds.containsF()));
        ListF<Long> deleteResourceIds = foundResourceIds.filter(resourceIds.containsF().notF()).toList();

        resourceReservationDao.insertBatch(creates);
        resourceReservationDao.updateByIdCreatorUidAndResourceId(updates);
        resourceReservationDao.deleteByIdCreatorUidAndResourceIds(reservationId, creatorUid, deleteResourceIds);

        return reservationId;
    }

    public void cancelReservations(PassportUid uid, long reservationId) {
        resourceReservationDao.deleteByIdAndCreatorUid(reservationId, uid);
    }

    public void deleteReservationsDeadlinedBefore(Instant before) {
        resourceReservationDao.deleteDeadlinedBefore(before);
    }

    public void cancelReservations(PassportUid uid, EventWithRelations event) {
        InstantInterval interval = new InstantInterval(event.getStartTs(), event.getEndTs());
        resourceReservationDao.findByResourceIdsAndIntervalAndCreatorUid(event.getResourceIds(), interval, uid)
                .stableUniqueBy(ResourceReservation::getReservationId)
                .forEach(info -> cancelReservations(uid, info.getReservationId()));
    }
}
