package ru.yandex.calendar.logic.resource.schedule;

import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.logic.beans.generated.ResourceReservation;
import ru.yandex.calendar.logic.event.IntervalComparator;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author dbrylev
 */
public class EventIdOrReservationInterval {
    public static final Comparator<EventIdOrReservationInterval> comparator =
            IntervalComparator.INSTANCE.compose(EventIdOrReservationInterval::getInterval)
                    .thenComparing(getReservationIdOrNullF().andThenNaturalComparator().nullLowC())
                    .thenComparing(getEventIdOrNullF().andThenNaturalComparator().nullLowC());

    private final Either<Long, ResourceReservation> eventIdOrReservation;
    private final InstantInterval interval;

    public EventIdOrReservationInterval(
            Either<Long, ResourceReservation> eventIdOrReservation, InstantInterval interval)
    {
        this.eventIdOrReservation = eventIdOrReservation;
        this.interval = interval;
    }

    public InstantInterval getInterval() {
        return interval;
    }

    public Either<Long, ResourceReservation> getEventIdOrReservation() {
        return eventIdOrReservation;
    }

    public Either<Long, Long> getEventIdOrReservationId() {
        return Either.fromOptions(getEventId(), getReservationId());
    }

    public boolean isEvent() {
        return eventIdOrReservation.isLeft();
    }

    public Option<Long> getEventId() {
        return eventIdOrReservation.leftO();
    }

    public Option<ResourceReservation> getReservation() {
        return eventIdOrReservation.rightO();
    }

    public Option<Long> getReservationId() {
        return eventIdOrReservation.rightO().map(ResourceReservation::getReservationId);
    }

    public Option<PassportUid> getReservationCreatorUid() {
        return eventIdOrReservation.rightO().map(ResourceReservation::getCreatorUid);
    }

    public static EventIdOrReservationInterval eventId(long eventId, InstantInterval interval) {
        return new EventIdOrReservationInterval(Either.left(eventId), interval);
    }

    public static EventIdOrReservationInterval reservation(ResourceReservation reservation, InstantInterval interval) {
        return new EventIdOrReservationInterval(Either.right(reservation), interval);
    }

    public static Function<EventIdOrReservationInterval, Long> getReservationIdOrNullF() {
        return e -> e.getReservationId().getOrNull();
    }

    public static Function<EventIdOrReservationInterval, Long> getEventIdOrNullF() {
        return e -> e.getEventId().getOrNull();
    }
}
