package ru.yandex.calendar.logic.resource.schedule;

import org.joda.time.Instant;

import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.logic.beans.generated.ResourceSchedule;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author dbrylev
 */
abstract class InvalidOrMissingSchedule {
    private final long resourceId;
    private final InstantInterval dayInterval;

    protected InvalidOrMissingSchedule(long resourceId, InstantInterval dayInterval) {
        this.resourceId = resourceId;
        this.dayInterval = dayInterval;
    }

    private static class Missing extends InvalidOrMissingSchedule {
        public Missing(long resourceId, InstantInterval dayInterval) {
            super(resourceId, dayInterval);
        }
    }

    private static class Invalid extends InvalidOrMissingSchedule {
        private final Instant scheduleDayStart;
        private final int scheduleVersion;

        public Invalid(long resourceId, InstantInterval dayInterval, Instant scheduleDayStart, int scheduleVersion) {
            super(resourceId, dayInterval);

            this.scheduleDayStart = scheduleDayStart;
            this.scheduleVersion = scheduleVersion;
        }
    }

    public static InvalidOrMissingSchedule missing(long resourceId, InstantInterval day) {
        return new Missing(resourceId, day);
    }

    public static InvalidOrMissingSchedule invalid(ResourceSchedule schedule, InstantInterval day) {
        Validate.isFalse(schedule.getIsValid());
        return new Invalid(schedule.getResourceId(), day, schedule.getDayStart(), schedule.getVersion());
    }

    public boolean isInvalid() {
        return !isMissing();
    }

    public boolean isMissing() {
        return this instanceof Missing;
    }

    public Instant getScheduleDayStart() {
        Validate.isTrue(isInvalid(), "Unknown id for missing schedule");
        return ((Invalid) this).scheduleDayStart;
    }

    public int getScheduleVersion() {
        Validate.isTrue(isInvalid(), "Unknown version for missing schedule");
        return ((Invalid) this).scheduleVersion;
    }

    public long getResourceId() {
        return resourceId;
    }

    public InstantInterval getDayInterval() {
        return dayInterval;
    }

    public static Function2<Long, InstantInterval, InvalidOrMissingSchedule> missingF() {
        return InvalidOrMissingSchedule::missing;
    }

    public static Function2<ResourceSchedule, InstantInterval, InvalidOrMissingSchedule> invalidF() {
        return InvalidOrMissingSchedule::invalid;
    }

    public static Function<InvalidOrMissingSchedule, InstantInterval> getDayIntervalF() {
        return new Function<InvalidOrMissingSchedule, InstantInterval>() {
            public InstantInterval apply(InvalidOrMissingSchedule s) {
                return s.getDayInterval();
            }
        };
    }

    public static Function<InvalidOrMissingSchedule, Long> getResourceIdF() {
        return new Function<InvalidOrMissingSchedule, Long>() {
            public Long apply(InvalidOrMissingSchedule s) {
                return s.getResourceId();
            }
        };
    }
}
