package ru.yandex.calendar.logic.resource.schedule;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.event.repetition.EventIndentInterval;
import ru.yandex.calendar.logic.resource.UidOrResourceId;

/**
 * @author dbrylev
 */
class LoadedDaySchedule {
    private final InvalidOrMissingSchedule schedule;
    private final ListF<EventIndentInterval> eventIntervals;

    public LoadedDaySchedule(InvalidOrMissingSchedule schedule, ListF<EventIndentInterval> eventIntervals) {
        this.schedule = schedule;
        this.eventIntervals = eventIntervals;
    }

    public ResourceScheduleData toData() {
        return new ResourceScheduleData(
                schedule.getResourceId(),
                eventIntervals.toTuple2List(EventIndentInterval::getEventId, EventIndentInterval::getInterval));
    }

    public static Function1B<LoadedDaySchedule> wasMissingF() {
        return new Function1B<LoadedDaySchedule>() {
            public boolean apply(LoadedDaySchedule s) {
                return s.getSchedule().isMissing();
            }
        };
    }

    public static Function1B<LoadedDaySchedule> wasInvalidF() {
        return new Function1B<LoadedDaySchedule>() {
            public boolean apply(LoadedDaySchedule s) {
                return s.getSchedule().isInvalid();
            }
        };
    }

    public static Function<LoadedDaySchedule, Long> getResourceIdF() {
        return new Function<LoadedDaySchedule, Long>() {
            public Long apply(LoadedDaySchedule s) {
                return s.getSchedule().getResourceId();
            }
        };
    }

    public static Function<LoadedDaySchedule, LocalDate> getDayDateF(final DateTimeZone tz) {
        return new Function<LoadedDaySchedule, LocalDate>() {
            public LocalDate apply(LoadedDaySchedule s) {
                return s.getSchedule().getDayInterval().getStart().toDateTime(tz).toLocalDate();
            }
        };
    }

    public static Function<LoadedDaySchedule, InvalidOrMissingSchedule> getScheduleF() {
        return new Function<LoadedDaySchedule, InvalidOrMissingSchedule>() {
            public InvalidOrMissingSchedule apply(LoadedDaySchedule s) {
                return s.getSchedule();
            }
        };
    }

    public InvalidOrMissingSchedule getSchedule() {
        return schedule;
    }

    public ListF<EventIndentInterval> getEventIntervals() {
        return eventIntervals;
    }

    public UidOrResourceId getResourceSubjectId() {
        return UidOrResourceId.resource(schedule.getResourceId());
    }

    public Instant getDayStart() {
        return schedule.getDayInterval().getStart();
    }
}
