package ru.yandex.calendar.logic.resource.schedule;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple4;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.logic.beans.GenericBeanDao;
import ru.yandex.calendar.logic.beans.generated.ResourceSchedule;
import ru.yandex.calendar.logic.beans.generated.ResourceScheduleFields;
import ru.yandex.calendar.logic.beans.generated.ResourceScheduleHelper;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlQueryUtils;

/**
 * @author gutman
 */
public class ResourceScheduleDao extends CalendarJdbcDaoSupport {

    @Autowired
    private GenericBeanDao genericBeanDao;

    public void insertResourceSchedulesIgnoreDuplicates(ListF<ResourceSchedule> schedules) {
        schedules = schedules.sorted(Comparator.<ResourceSchedule>constEqualComparator()
                .thenComparing(ResourceSchedule::getResourceId)
                .thenComparing(ResourceSchedule::getDayStart));

        genericBeanDao.insertBeansIgnoreDuplicates(schedules);
    }

    @RunWithRandomTest
    public void updateResourceSchedulesSetNotValidByResourceIds(ListF<Long> resourceIds) {
        SqlCondition c = ResourceScheduleFields.RESOURCE_ID.column().inSet(resourceIds)
                .and(ResourceScheduleFields.IS_VALID.eq(true));

        String q = "UPDATE resource_schedule rs SET is_valid = FALSE, version = version + 1"
                + " FROM (SELECT resource_id, day_start FROM resource_schedule"
                + c.whereSql() + " ORDER BY resource_id, day_start FOR NO KEY UPDATE) x"
                + " WHERE rs.resource_id = x.resource_id AND rs.day_start = x.day_start";

        if (skipQuery(c, q, c.args())) return;

        getJdbcTemplate().update(q, c.args());
    }

    @RunWithRandomTest
    public void updateResourceSchedulesSetIsValidAndIntervals(
            ListF<Tuple4<String, Long, Instant, Integer>> schedules)
    {
        String q = "UPDATE resource_schedule"
                + " SET event_intervals = ?, is_valid = TRUE, version = version + 1"
                + " WHERE resource_id = ? AND day_start = ? AND version = ?";

        schedules = schedules.sorted(Comparator.<Tuple4<String, Long, Instant, Integer>>constEqualComparator()
                .thenComparing(Tuple4::get2).thenComparing(Tuple4::get3));

        genericBeanDao.batchUpdate(q, schedules.map(t -> Cf.list(t.get1(), t.get2(), t.get3(), t.get4())));
    }

    @RunWithRandomTest
    public ListF<ResourceSchedule> findResourceSchedulesByResourceIdsAndDays(
            ListF<Long> resourceIds, ListF<Instant> daysStarts)
    {
        SqlCondition c = ResourceScheduleFields.RESOURCE_ID.column().inSet(resourceIds)
                .and(ResourceScheduleFields.DAY_START.column().inSet(daysStarts));

        String q = "SELECT * FROM resource_schedule" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, ResourceSchedule.class, c.args());
    }

    @RunWithRandomTest
    public void deleteResourceSchedulesByResourceIds(ListF<Long> resourceIds) {
        String q = "DELETE FROM resource_schedule WHERE resource_id " + SqlQueryUtils.inSet(resourceIds);

        if (skipQuery(resourceIds, q)) return;

        getJdbcTemplate().update(q);
    }

    @RunWithRandomTest
    public void deleteResourceSchedulesStartedBefore(Instant before) {
        genericBeanDao.deleteBeans(ResourceScheduleHelper.INSTANCE, ResourceScheduleFields.DAY_START.lt(before));
    }

    @RunWithRandomTest
    public void deleteAllResourceSchedules() {
        String q = "DELETE FROM resource_schedule";
        getJdbcTemplate().update(q);
    }

}
