package ru.yandex.calendar.logic.resource.schedule;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.beans.generated.ResourceSchedule;
import ru.yandex.calendar.logic.suggest.IntervalSet;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

/**
* @author gutman
*/
public class ResourceScheduleData {

    private final long resourceId;
    private final Tuple2List<Long, InstantInterval> eventIntervals;

    ResourceScheduleData(long resourceId, Tuple2List<Long, InstantInterval> eventIntervals) {
        this.resourceId = resourceId;
        this.eventIntervals = eventIntervals;
    }

    static ResourceScheduleData parse(ResourceSchedule rs) {
        return new ResourceScheduleData(rs.getResourceId(),
                ResourceScheduleManager.eventIntervalsFromString(rs.getEventIntervals()));
    }

    static Function<ResourceSchedule, ResourceScheduleData> parseF() {
        return ResourceScheduleData::parse;
    }

    public static ResourceScheduleData merge(ListF<ResourceScheduleData> schedules) {
        Validate.hasSize(1, schedules.map(ResourceScheduleData.getResourceIdF()).unique());

        ListF<Tuple2<Long, InstantInterval>> intervals = Cf.arrayList();
        for (ResourceScheduleData schedule : schedules) {
            intervals.addAll(schedule.getEventIntervals());
        }
        intervals = intervals.stableUnique();

        return new ResourceScheduleData(schedules.first().getResourceId(), Tuple2List.tuple2List(intervals));
    }

    public ResourceScheduleData intersect(InstantInterval interval) {
        return new ResourceScheduleData(
                getResourceId(), getEventIntervals().filterBy2(i -> i.overlaps(interval)));
    }

    public long getResourceId() {
        return resourceId;
    }

    public Tuple2List<Long, InstantInterval> getEventIntervals() {
        return eventIntervals;
    }

    public ListF<InstantInterval> getInstantIntervals() {
        return eventIntervals.get2();
    }

    public static Function<ResourceScheduleData, ListF<Long>> getEventIdsF() {
        return new Function<ResourceScheduleData, ListF<Long>>() {
            public ListF<Long> apply(ResourceScheduleData parsedResourceSchedule) {
                return parsedResourceSchedule.eventIntervals.get1();
            }
        };
    }

    public static Function<ResourceScheduleData, ListF<InstantInterval>> getInstantIntervalsF() {
        return new Function<ResourceScheduleData, ListF<InstantInterval>>() {
            public ListF<InstantInterval> apply(ResourceScheduleData s) {
                return s.eventIntervals.get2();
            }
        };
    }

    public static Function<ResourceScheduleData, Long> getResourceIdF() {
        return new Function<ResourceScheduleData, Long>() {
            public Long apply(ResourceScheduleData resourceScheduleData) {
                return resourceScheduleData.getResourceId();
            }
        };
    }

    public static Function<ResourceScheduleInfo, ResourceScheduleData> fromResourceScheduleInfoF() {
        return new Function<ResourceScheduleInfo, ResourceScheduleData>() {
            public ResourceScheduleData apply(ResourceScheduleInfo i) {
                return new ResourceScheduleData(i.getResource().getResource().getId(),
                        i.getEvents().zipWith(EventIntervalInfo.getIntervalF()).map1(EventIntervalInfo.getEventIdF()));
            }
        };
    }

    public static Function<ListF<ResourceScheduleData>, ResourceScheduleData> mergeF() {
        return new Function<ListF<ResourceScheduleData>, ResourceScheduleData>() {
            public ResourceScheduleData apply(ListF<ResourceScheduleData> rs) {
                return merge(rs);
            }
        };
    }

    public static Function<ResourceScheduleData, ResourceScheduleData> intersectF(final InstantInterval interval) {
        return new Function<ResourceScheduleData, ResourceScheduleData>() {
            public ResourceScheduleData apply(ResourceScheduleData s) {
                return s.intersect(interval);
            }
        };
    }

    public static Function<ResourceScheduleData, ListF<InstantInterval>> getMergedIntervalsF() {
        return new Function<ResourceScheduleData, ListF<InstantInterval>>() {
            public ListF<InstantInterval> apply(ResourceScheduleData s) {
                return IntervalSet.cons(s.eventIntervals.get2()).getIntervals();
            }
        };
    }

}
