package ru.yandex.calendar.logic.resource.schedule;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import lombok.val;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.Tuple4;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.logic.beans.generated.ResourceSchedule;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventInfoDbLoader;
import ru.yandex.calendar.logic.event.EventLoadLimits;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.repetition.EventIndentAndRepetition;
import ru.yandex.calendar.logic.event.repetition.EventIndentInterval;
import ru.yandex.calendar.logic.event.repetition.InfiniteInterval;
import ru.yandex.calendar.logic.event.repetition.IntersectingIntervals;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceSet;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.reservation.ResourceReservationInfo;
import ru.yandex.calendar.logic.resource.reservation.ResourceReservationManager;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.suggest.SuggestUtils;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.InstantIntervalSet;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.net.AssertNoNetworkAccessStack;
import ru.yandex.misc.net.AssertNoNetworkAccessStackHandle;
import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.Iso8601;

public class ResourceScheduleManager {
    private static final Logger logger = LoggerFactory.getLogger(ResourceScheduleManager.class);

    @Autowired
    private ResourceScheduleDao resourceScheduleDao;
    @Autowired
    private ResourceReservationManager resourceReservationManager;
    @Autowired
    private EventInfoDbLoader eventInfoDbLoader;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private UserManager userManager;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private TransactionTemplate transactionTemplate;
    @Autowired
    private EnvironmentType environmentType;

    public ResourceScheduleInfo getResourceScheduleForDayAndResource(
            PassportUid uid, LocalDate day, DateTimeZone tz, ResourceInfo resource, ActionSource actionSource)
    {
        ResourceScheduleData schedule = getCachedSchedulesAndComputeMissingInCache(
                Cf.list(resource.getResourceId()), Cf.list(day), tz).get2().single().single();

        return toResourceScheduleInfo(uid, Cf.list(resource), Cf.list(schedule), actionSource).single();
    }

    public Tuple2List<ResourceInfo, InstantIntervalSet> getResourceBusyIntervals(
            Option<PassportUid> uid, ListF<ResourceInfo> resources,
            Instant start, Instant end, DateTimeZone tz, ActionInfo actionInfo)
    {
        InstantInterval interval = new InstantInterval(start, end);

        return getResourceScheduleDataForInterval(uid, resources, interval, tz, Option.empty(), actionInfo)
                .map2(InstantIntervalSet.unionF().compose(ResourceEventsAndReservations::getInstantIntervals));
    }

    public Tuple2List<ResourceInfo, ResourceEventsAndReservations> getResourceScheduleDataForInterval(
            Option<PassportUid> uid, ListF<ResourceInfo> resources,
            InstantInterval interval, DateTimeZone tz, Option<Long> exceptEventId, ActionInfo actionInfo)
    {
        Function<Long, ResourceInfo> resourceByIdF = resources.toMapMappingToKey(ResourceInfo.resourceIdF())::getOrThrow;
        Function<ResourceEventsAndReservations, ResourceEventsAndReservations> excludeEventsF =
                excludeEventsFromScheduleF(exceptEventId);

        return getEventsAndReservations(uid, resources, interval, tz, actionInfo).toTuple2List(
                resourceByIdF.compose(ResourceEventsAndReservations::getResourceId),
                excludeEventsF);
    }

    public ListF<ResourceDaySchedule> getResourceScheduleDataForDays(
            Option<PassportUid> exceptUid, ListF<Long> resourceIds,
            ListF<LocalDate> days, DateTimeZone tz, ListF<Long> exceptEventIds, ActionInfo actionInfo)
    {
        if (days.isEmpty() || resourceIds.isEmpty()) return Cf.list();

        Function<ResourceEventsAndReservations, ResourceEventsAndReservations> excludeEventsF =
                excludeEventsFromScheduleF(exceptEventIds);

        ListF<ResourceDaySchedule> result = Cf.arrayList();

        getEventsAndReservations(exceptUid, resourceIds, days, tz, actionInfo).forEach((date, schedules) ->
                schedules.forEach(s -> result.add(new ResourceDaySchedule(date, excludeEventsF.apply(s)))));

        return result;
    }

    private ListF<ResourceEventsAndReservations> getEventsAndReservations(
            Option<PassportUid> exceptUid, ListF<ResourceInfo> resources,
            InstantInterval interval, DateTimeZone tz, ActionInfo actionInfo)
    {
        ListF<LocalDate> days = AuxDateTime.splitByDays(interval, tz)
                .map(AuxDateTime.instantDateF(tz).compose(InstantInterval::getStart));

        Tuple2List<LocalDate, ListF<ResourceEventsAndReservations>> schedules = getEventsAndReservations(
                exceptUid, resources.map(ResourceInfo.resourceIdF()), days, tz, actionInfo);

        ListF<ResourceEventsAndReservations> cropped = schedules.flatMap(t -> {
            if (t.get1().equals(days.first()) || t.get1().equals(days.last())) {
                return t.get2().map(r -> r.filterIntervals(i -> i.getInterval().overlaps(interval)));
            } else {
                return t.get2();
            }
        });
        CollectionF<ListF<ResourceEventsAndReservations>> grouped = cropped
                .groupBy(ResourceEventsAndReservations::getResourceId).values();

        return grouped.map(es -> {
            ListF<EventIdOrReservationInterval> intervals = es.flatMap(ResourceEventsAndReservations::getIntervals);
            return new ResourceEventsAndReservations(
                    es.first().getResourceId(),
                    intervals.stableUniqueBy(i -> Tuple2.tuple(i.getInterval(), i.getEventIdOrReservationId())));
        });
    }

    private Tuple2List<LocalDate, ListF<ResourceEventsAndReservations>> getEventsAndReservations(
            Option<PassportUid> exceptUid, ListF<Long> resourceIds,
            ListF<LocalDate> dates, DateTimeZone tz, ActionInfo actionInfo)
    {
        Tuple2List<LocalDate, ListF<ResourceScheduleData>> events =
                getCachedSchedulesAndComputeMissingInCache(resourceIds, dates, tz);

        MapF<Tuple2<LocalDate, Long>, ListF<EventIdOrReservationInterval>> reservations =
                loadReservations(exceptUid, resourceIds, dates, tz, actionInfo);

        Tuple2List<LocalDate, ListF<ResourceEventsAndReservations>> result = Tuple2List.arrayList();

        events.forEach((date, schedules) -> result.add(date, schedules.map(schedule -> {
            ListF<EventIdOrReservationInterval> eventIntervals = schedule.getEventIntervals()
                    .map(EventIdOrReservationInterval::eventId);

            Option<ListF<EventIdOrReservationInterval>> reserveIntervals =
                    reservations.getO(Tuple2.tuple(date, schedule.getResourceId()));

            return new ResourceEventsAndReservations(schedule.getResourceId(), reserveIntervals.isPresent()
                    ? reserveIntervals.get().plus(eventIntervals).sorted(EventIdOrReservationInterval.comparator)
                    : eventIntervals);
        })));
        return result;
    }

    private MapF<Tuple2<LocalDate, Long>, ListF<EventIdOrReservationInterval>> loadReservations(
            Option<PassportUid> exceptUid, ListF<Long> resourceIds,
            ListF<LocalDate> dates, DateTimeZone tz, ActionInfo actionInfo)
    {
        if (dates.isEmpty()) return Cf.map();

        InstantInterval interval = new InstantInterval(
                dates.map(d -> d.toDateTimeAtStartOfDay(tz).toInstant()).min(),
                dates.map(d -> d.plusDays(1).toDateTimeAtStartOfDay(tz).toInstant()).max());

        RepetitionInstanceSet daysSet = RepetitionInstanceSet.fromSuccessiveIntervals(
                dates.stableUnique().map(AuxDateTime.dayIntervalF(tz)).sortedBy(InstantInterval::getStart));

        MapF<Tuple2<LocalDate, Long>, ListF<EventIdOrReservationInterval>> result = Cf.hashMap();
        for (ResourceReservationInfo reservation
                : resourceReservationManager.findReservations(resourceIds, interval, exceptUid, actionInfo))
        {
            for (IntersectingIntervals intersection : daysSet.overlap(
                    new RepetitionInstanceSet(reservation.getRepetitionInfo(), interval.getStart(), interval.getEnd())))
            {
                EventIdOrReservationInterval res = EventIdOrReservationInterval.reservation(
                        reservation.getReservation(), intersection.getSecondInterval());

                LocalDate date = new LocalDate(intersection.getFirstInterval().getStart(), tz);
                result.getOrElseUpdate(Tuple2.tuple(date, reservation.getResourceId()), Cf::arrayList).add(res);
            }
        }
        return result;
    }

    private Tuple2List<LocalDate, ListF<ResourceScheduleData>> getCachedSchedulesAndComputeMissingInCache(
            ListF<Long> resourceIds, ListF<LocalDate> dates, DateTimeZone tz)
    {
        ListF<InstantInterval> days = dates.map(AuxDateTime.dayIntervalF(tz)).stableUnique();

        ListF<ResourceSchedule> found = resourceScheduleDao.findResourceSchedulesByResourceIdsAndDays(
                resourceIds, days.map(InstantInterval::getStart));

        Function1B<ResourceSchedule> isValidF = Function1B.wrap(ResourceSchedule.getIsValidF());
        Function<ResourceSchedule, InstantInterval> dayIntervalF = ResourceSchedule.getDayStartF()
                .andThen(Cf2.f(days.toMapMappingToKey(InstantInterval::getStart)::getOrThrow));

        ListF<ResourceSchedule> valid = found.filter(isValidF);

        ListF<InvalidOrMissingSchedule> invalid = found.filter(isValidF.notF()).zipWith(dayIntervalF)
                .map(InvalidOrMissingSchedule.invalidF());

        ListF<InvalidOrMissingSchedule> missing = Cf2.join(resourceIds, days).unique()
                .minus(found.toTuple2List(ResourceSchedule.getResourceIdF(), dayIntervalF))
                .map(InvalidOrMissingSchedule.missingF().asFunction());

        ListF<LoadedDaySchedule> loaded = MasterSlaveContextHolder.withPolicy(
                MasterSlavePolicy.R_SYNC_SM, () -> loadResourceSchedules(invalid.plus(missing)));

        if (environmentType != EnvironmentType.TESTS) {
            storeInCacheAsynchronously(loaded);
        } else {
            storeInCache(loaded);
        }

        MapF<LocalDate, ListF<ResourceScheduleData>> grouped = Cf.hashMap();

        valid.forEach(s -> grouped.getOrElseUpdate(new LocalDate(s.getDayStart(), tz), Cf::arrayList)
                .add(ResourceScheduleData.parse(s)));

        loaded.forEach(s -> grouped.getOrElseUpdate(new LocalDate(s.getDayStart(), tz), Cf::arrayList)
                .add(s.toData()));

        return dates.zipWith(localDate -> grouped.getOrDefault(localDate, Cf.arrayList()));
    }

    private ListF<LoadedDaySchedule> loadResourceSchedules(ListF<InvalidOrMissingSchedule> schedules) {
        if (schedules.isEmpty()) return Cf.list();

        MapF<Long, ListF<InvalidOrMissingSchedule>> successiveSchedulesByResourceId = schedules
                .sortedBy(s -> s.getDayInterval().getStart())
                .groupBy(InvalidOrMissingSchedule::getResourceId);

        ListF<InstantInterval> allDays = schedules.map(InvalidOrMissingSchedule::getDayInterval);

        Instant start = allDays.iterator().map(InstantInterval::getStart).min();
        Instant end = allDays.iterator().map(InstantInterval::getEnd).max();

        InfiniteInterval interval = new InfiniteInterval(start, Option.of(end));

        ListF<EventIndentAndRepetition> resourcesEvents = eventInfoDbLoader.getEventIndentsOnResources(
                successiveSchedulesByResourceId.keys(), EventLoadLimits.intersectsInterval(interval));

        ListF<EventIndentInterval> resourceEventIntervals = resourcesEvents
                .flatMap(EventIndentAndRepetition.getInstancesInIntervalF(interval))
                .sorted(EventIndentInterval.comparator);

        MapF<InvalidOrMissingSchedule, ListF<EventIndentInterval>> result = schedules
                .toMap(s -> s, s -> Cf.arrayList());

        resourceEventIntervals.forEach(ri -> {
            long resourceId = ri.getIndent().getLayerOrResourceId();

            ListF<InvalidOrMissingSchedule> overlapped = SuggestUtils.findOverlappingForSuccessiveIntervals(
                    successiveSchedulesByResourceId.getOrThrow(resourceId),
                    InvalidOrMissingSchedule.missing(resourceId, ri.getInterval()),
                    InvalidOrMissingSchedule::getDayInterval);

            overlapped.forEach(o -> result.getOrThrow(o).add(ri));
        });

        return result.entrySet().map(e -> new LoadedDaySchedule(e.getKey(), e.getValue()));
    }

    private ListF<ResourceScheduleInfo> toResourceScheduleInfo(final PassportUid uid, ListF<ResourceInfo> resources,
                                                               ListF<ResourceScheduleData> parsed, ActionSource actionSource) {
        val user = userManager.getUserInfo(uid);
        val allEventIds = parsed.flatMap(ResourceScheduleData.getEventIdsF());

        val eventsWithRelations = eventDbManager.getEventsWithRelationsByIds(allEventIds);
        val eventsWithRelationsByEventId = eventsWithRelations.toMapMappingToKey(EventWithRelations::getId);
        val eventAuthInfoById = authorizer.loadEventsInfoForPermsCheck(user, eventsWithRelations);

        val resourcesById = resources.toMapMappingToKey(ResourceInfo.resourceIdF());

        authorizer.loadBatchAndCacheAllRequiredForPermsCheck(uid, eventsWithRelations);

        AssertNoNetworkAccessStackHandle h = AssertNoNetworkAccessStack.push();
        try {
            return parsed.flatMap(parsedResourceSchedule -> {
                if (!resourcesById.containsKeyTs(parsedResourceSchedule.getResourceId())) {
                    return Option.empty(); // resource may be turned off already
                }

                Function2<Long, InstantInterval, Option<EventIntervalInfo>> toEventIntervalInfoF =
                    (eventId, interval) -> {
                        if (!eventsWithRelationsByEventId.containsKeyTs(eventId)) {
                            logger.warn("Event from cache not found in db " + eventId);
                            return Option.empty();
                        }

                        val event = eventsWithRelationsByEventId.getTs(eventId);
                        val eventAuthInfo = eventAuthInfoById.get(eventId);
                        val canView = authorizer.canViewEvent(user, eventAuthInfo, actionSource);
                        return Option.of(new EventIntervalInfo(interval, event, canView));
                    };

                ListF<EventIntervalInfo> eventIntervalInfos =
                        parsedResourceSchedule.getEventIntervals().map(toEventIntervalInfoF).flatten();
                ResourceInfo resource = resourcesById.getTs(parsedResourceSchedule.getResourceId());

                return Option.of(new ResourceScheduleInfo(resource, eventIntervalInfos));
            });
        } finally {
            h.popSafely();
        }
    }

    public void invalidateCachedScheduleForResources(ListF<Long> resourceIds) {
        resourceScheduleDao.updateResourceSchedulesSetNotValidByResourceIds(resourceIds.sorted());
    }

    // for adminka
    public void deleteAllCachedSchedules() {
        resourceScheduleDao.deleteAllResourceSchedules();
    }

    private final ExecutorService executor = Executors.newFixedThreadPool(3);

    private void storeInCacheAsynchronously(final ListF<LoadedDaySchedule> rs) {
        if (rs.isEmpty()) return;

        executor.submit(() -> {
            WhatThreadDoes.Handle wtdHandle = WhatThreadDoes.push("stores resources schedule into cache");
            RequestIdStack.Handle ridHandle = RequestIdStack.pushIfNotYet();
            try {
                storeInCache(rs);

            } catch (Exception e) {
                logger.warn("Can not save resource schedule for resources " +
                        rs.map(LoadedDaySchedule.getResourceIdF()).stableUnique(), e);

            } finally {
                wtdHandle.popSafely();
                ridHandle.popSafely();
            }
        });
    }

    private void storeInCache(ListF<LoadedDaySchedule> rs) {
        IteratorF<LoadedDaySchedule> invalid = rs.iterator().filter(LoadedDaySchedule.wasInvalidF());
        IteratorF<LoadedDaySchedule> missing = rs.iterator().filter(LoadedDaySchedule.wasMissingF());

        invalid.paginate(1000).forEachRemaining(ss ->
                resourceScheduleDao.updateResourceSchedulesSetIsValidAndIntervals(ss.map(s -> Tuple4.tuple(
                        eventIntervalsToString(s.getEventIntervals()),
                        s.getSchedule().getResourceId(),
                        s.getSchedule().getScheduleDayStart(),
                        s.getSchedule().getScheduleVersion()))));

        if (!missing.hasNext()) return;

        Function0V insert = () -> {
            SetF<Long> lockingIds = rs.iterator().filter(s -> s.getSchedule().isMissing())
                    .map(s -> s.getSchedule().getResourceId()).toSet();

            SetF<Long> lockedIds = resourceRoutines.tryLockResourcesByIds(lockingIds.toList()).unique();

            missing.paginate(1000).forEachRemaining(ss -> {
                ss = ss.filter(s -> lockedIds.containsTs(s.getSchedule().getResourceId()));

                resourceScheduleDao.insertResourceSchedulesIgnoreDuplicates(ss.map(s -> {
                    ResourceSchedule resourceSchedule = new ResourceSchedule();
                    resourceSchedule.setResourceId(s.getSchedule().getResourceId());
                    resourceSchedule.setDayStart(s.getSchedule().getDayInterval().getStart());

                    resourceSchedule.setEventIntervals("");
                    resourceSchedule.setVersion(0);
                    resourceSchedule.setIsValid(false);

                    return resourceSchedule;

                }));
            });
        };

        if (environmentType != EnvironmentType.TESTS) {
            transactionTemplate.execute(insert.asFunctionReturnParam()::apply);

        } else {
            insert.apply();
        }
    }

    // id1(start1,end1);id2(start2,end2);...
    public static Tuple2List<Long, InstantInterval> eventIntervalsFromString(String eventIntervals) {
        return Tuple2List.tuple2List(Cf.x(eventIntervals.split(";")).filter(Cf.String.notEmptyF()).map(
            s -> {
                String[] parts = s.split("\\(");
                long eventId = Long.parseLong(parts[0]);
                String[] startEnd = parts[1].substring(0, parts[1].length() - 1).split(",");
                Instant start = Iso8601.parseDateTime(startEnd[0]).toInstant();
                Instant end = Iso8601.parseDateTime(startEnd[1]).toInstant();
                return Tuple2.tuple(eventId, new InstantInterval(start, end));
            }));
    }

    private static String eventIntervalsToString(ListF<EventIndentInterval> eventIntervals) {
        return eventIntervals.map(
                info -> info.getEventId() + "(" + info.getInterval().getStart() + "," + info.getInterval().getEnd() + ")"
        ).mkString(";");
    }

    private Function<ResourceEventsAndReservations, ResourceEventsAndReservations> excludeEventsFromScheduleF(
            ListF<Long> exceptEventIds)
    {
        ListF<Long> eventIds = exceptEventIds.isNotEmpty()
                ? eventRoutines.findMasterAndSingleEventIds(exceptEventIds)
                : Cf.list();

        if (eventIds.isEmpty()) return Function.identityF();

        return s -> s.filterIntervals(i -> !i.getEventId().exists(eventIds.containsF()));
    }
}
