package ru.yandex.calendar.logic.sending;

import lombok.Getter;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.contact.UnivContact;
import ru.yandex.calendar.logic.event.ChangedEventInfoForMails;
import ru.yandex.calendar.logic.ics.exp.EventInstanceParameters;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.sending.param.MessageDestination;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.MailType;
import ru.yandex.calendar.logic.sharing.participant.UserParticipantInfo;
import ru.yandex.misc.lang.DefaultToString;
import ru.yandex.misc.lang.Validate;

/**
 * A structure that contains information per invited user
 * (information is about: object, sender, recipient,
 * mail type, additional information)
 */
@Getter
public class EventSendingInfo extends DefaultToString {
    // Required fields
    private final long eventId;
    private final boolean omitIcs;
    private final UnivContact univContact;
    private final Option<String> privateTokenO;
    private final MailType meetingMailType;

    private final Decision decision;

    private final EventInstanceParameters eventInstanceParameters;
    private final Option<ChangedEventInfoForMails> changesInfo;

    private final Option<Long> reorganizedEventId;
    private final Option<ResourceInfo> subscriptionResource;

    private final MessageDestination destination;

    public EventSendingInfo(
            UserParticipantInfo userParticipantInfo, MailType meetingMailType,
            EventInstanceParameters eventInstanceParameters, Option<ChangedEventInfoForMails> changesInfo)
    {
        this(
                UnivContact.formParticipantInfo(userParticipantInfo),
                userParticipantInfo.getPrivateToken(),
                userParticipantInfo.getEventId(),
                false, meetingMailType, eventInstanceParameters,
                changesInfo, userParticipantInfo.getDecision(), MessageDestination.ANYWHERE);
    }

    public EventSendingInfo(
            UnivContact univContact, Option<String> privateTokenO, long eventId, MailType meetingMailType,
            EventInstanceParameters eventInstanceParameters, Decision decision)
    {
        this(univContact, privateTokenO, eventId, false, meetingMailType,
                eventInstanceParameters, Option.empty(), decision, MessageDestination.ANYWHERE);
    }

    public EventSendingInfo(
            UnivContact univContact, Option<String> privateTokenO,
            long eventId, boolean omitIcs, MailType meetingMailType,
            EventInstanceParameters eventInstanceParameters,
            Option<ChangedEventInfoForMails> changesInfo, Decision decision,
            MessageDestination destination)
    {
        this(univContact, privateTokenO, eventId, omitIcs, meetingMailType,
                eventInstanceParameters, changesInfo, decision, Option.empty(), Option.empty(), destination);
    }

    public EventSendingInfo(
            UnivContact univContact, Option<String> privateTokenO,
            long eventId, boolean omitIcs, MailType meetingMailType,
            EventInstanceParameters eventInstanceParameters,
            Option<ChangedEventInfoForMails> changesInfo, Decision decision,
            Option<Long> reorganizedEventId, Option<ResourceInfo> subscriptionResource, MessageDestination destination)
    {
        Validate.isTrue((meetingMailType != MailType.EVENT_UPDATE) || changesInfo.isPresent());

        Validate.notNull(univContact);
        Validate.notNull(privateTokenO);
        Validate.notNull(meetingMailType);
        Validate.notNull(eventInstanceParameters);
        Validate.notNull(decision);

        this.univContact = univContact;
        this.privateTokenO = privateTokenO;
        this.eventId = eventId;
        this.omitIcs = omitIcs;
        this.meetingMailType = meetingMailType;
        this.eventInstanceParameters = eventInstanceParameters;
        this.changesInfo = changesInfo;
        this.decision = decision;
        this.reorganizedEventId = reorganizedEventId;
        this.subscriptionResource = subscriptionResource;
        this.destination = destination;
    }

    public EventSendingInfo reorganized(
            MailType type, ChangedEventInfoForMails changes, Option<Long> reorganizedEventId)
    {
        return new EventSendingInfo(
                univContact, privateTokenO, eventId, omitIcs,
                type, eventInstanceParameters, Option.of(changes),
                decision, reorganizedEventId, subscriptionResource, destination);
    }

    public EventSendingInfo withDestination(MessageDestination destination) {
        return new EventSendingInfo(univContact, privateTokenO, eventId, omitIcs, meetingMailType,
                eventInstanceParameters, changesInfo, decision, reorganizedEventId, subscriptionResource, destination);
    }

    public EventSendingInfo withSubscriptionResource(ResourceInfo resource) {
        return new EventSendingInfo(univContact, privateTokenO, eventId, omitIcs, meetingMailType,
                eventInstanceParameters, changesInfo, decision, reorganizedEventId, Option.of(resource), destination);
    }

    public static Function<EventSendingInfo, Long> getEventIdF() {
        return EventSendingInfo::getEventId;
    }

    public static Function1B<EventSendingInfo> isMeetingMailTypeInF(MailType... mailType) {
        return a -> Cf.set(mailType).containsTs(a.meetingMailType);
    }
}
