package ru.yandex.calendar.logic.sending.bazinga;

import java.time.Duration;
import java.time.Instant;

import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.postgresql.util.PGobject;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.sending.param.MessageExtra;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.parse.BenderParser;
import ru.yandex.misc.bender.serialize.BenderSerializer;

@Slf4j
public class MessageExtraDao extends CalendarJdbcDaoSupport {
    private final DynamicProperty<Long> defaultTtlMinutes = new DynamicProperty<>("messageExtraDefaultTtlMinutes", Duration.ofDays(3).toMinutes());

    public static final BenderSerializer<MessageExtra> serializer = Bender.serializer(
            MessageExtra.class,
            MessageParametersBender.getConfiguration()
    );
    public static final BenderParser<MessageExtra> parser = Bender.parser(
            MessageExtra.class,
            MessageParametersBender.getConfiguration()
    );

    public long save(MessageExtra messageExtra) {
        return save(messageExtra, Duration.ofMinutes(defaultTtlMinutes.get()));
    }

    @SneakyThrows
    public long save(MessageExtra messageExtra, Duration ttl) {
        var data = new PGobject() {
            public String toString() {
                return "MessageExtra[length=" + getValue().length() + "]";
            }
        };
        data.setType("json");
        data.setValue(new String(serializer.serializeJson(messageExtra)));
        var expirationTs = Instant.now().plus(ttl);
        return getJdbcTemplate().queryForLong("INSERT INTO message_extra (data, expiration_ts) VALUES (?, ?) RETURNING id", data, expirationTs);
    }

    public Option<MessageExtra> load(long id) {
        var result = getJdbcTemplate().queryForOption(
                "SELECT data FROM message_extra WHERE id = ?",
                (rs, rowNum) -> parser.parseJson(((PGobject) rs.getObject("data")).getValue()),
                id
        );
        return result;
    }

    @RunWithRandomTest
    public int deleteExpired() {
        return getJdbcTemplate().update(
                "DELETE FROM message_extra WHERE expiration_ts < ?", Instant.now());
    }
}
