package ru.yandex.calendar.logic.sending.param;

import org.jdom.Element;
import org.joda.time.LocalDateTime;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.commune.mail.MailAddress;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.annotation.FillWithSomething;
import ru.yandex.misc.annotation.SampleValue;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;

@BenderBindAllFields
public class CommonEventMessageParameters {

    private final Language language;
    private final LocalDateTime now;

    private final Sender sender;
    private final Recipient recipient;
    private final Email emailFrom;

    private final String calendarUrl;

    private final Option<Boolean> hideActor;
    private final Option<MessageOverrides> overrides;

    @FillWithSomething
    public CommonEventMessageParameters(
            Language language, LocalDateTime now,
            Sender sender, Recipient recipient, Email emailFrom,
            @SampleValue("http://calendar.yandex-team.ru/") String calendarUrl,
            boolean hideActor, MessageOverrides overrides)
    {
        this.language = language;
        this.now = now;
        this.sender = sender;
        this.recipient = recipient;
        this.emailFrom = emailFrom;
        this.calendarUrl = calendarUrl.replaceFirst("/$", "");
        this.hideActor = Option.of(hideActor);
        this.overrides = Option.of(overrides).filterNot(MessageOverrides::isEmpty);
    }

    public void appendXmlTo(Element r) {
        CalendarXmlizer.appendElm(r, "language", language.value());
        CalendarXmlizer.appendElm(r, "date-time-now", now.toString());

        CalendarXmlizer.appendElm(r, "to", recipient.address.getEmail().getEmail());
        CalendarXmlizer.appendElm(r, "nick", StringUtils.defaultIfEmpty(
                recipient.address.getPersonal(), Emails.getUnicoded(recipient.address.getEmail())));

        CalendarXmlizer.appendElm(r, "sender-nick", StringUtils.defaultIfEmpty(
                sender.getName().getName(language), Emails.getUnicoded(sender.getEmail())));
        CalendarXmlizer.appendElm(r, "sender-email", sender.getEmail().getEmail());
        CalendarXmlizer.appendElm(r, "sender-gender", sender.getGender()
                .map(g -> g.name().substring(0, 1)).getOrElse(""));

        CalendarXmlizer.appendElm(r, "email-from", emailFrom.getEmail());

        CalendarXmlizer.appendElm(r, "for-sender", recipient.address.getEmail().equalsIgnoreCase(sender.getEmail()));
        CalendarXmlizer.appendElm(r, "calendar-url", calendarUrl);

        if (recipient.uid.isPresent())
            CalendarXmlizer.appendElm(r, "uid", recipient.uid.get().getUid());

        hideActor.forEach(h -> CalendarXmlizer.appendElm(r, "hide-actor", h));
    }

    public Language getLanguage() {
        return language;
    }

    public Sender getSender() {
        return sender;
    }

    public Recipient getRecipient() {
        return recipient;
    }

    public MailAddress getRecipientAddress() {
        return recipient.address;
    }

    public Option<PassportUid> getRecipientUid() {
        return recipient.uid;
    }

    public LocalDateTime getNow() {
        return now;
    }

    public MessageOverrides getOverrides() {
        return overrides.getOrElse(MessageOverrides.EMPTY);
    }

    public String getCalendarUrl() {
        return calendarUrl;
    }

    public CommonEventMessageParameters withHiddenActor() {
        return new CommonEventMessageParameters(
                language, now, sender, recipient, emailFrom, calendarUrl, true, getOverrides());
    }

    public CommonEventMessageParameters withOverrides(MessageOverrides overrides) {
        return new CommonEventMessageParameters(
                language, now, sender, recipient, emailFrom, calendarUrl, hideActor.getOrElse(false), overrides);
    }

} //~
