package ru.yandex.calendar.logic.sending.param;

import org.jdom.Element;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceType;
import ru.yandex.calendar.logic.sending.EventXmlCreator;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.util.resources.UStringLiteral;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.lang.StringUtils;

@BenderBindAllFields
public class EventLocation {

    private static final ListF<String> ROAD_MAP_OFFICES = Cf.list(
            "avrora",
            "redrose",
            "rrs",
            "rrm",
            "mskoko",
            "lotteplaza",
            "amalteya",
            "spb",
            "eburg",
            "nsk",
            "kzn",
            "Innopolis",
            "rnd",
            "nnlob",
            "simf",
            "freshlaza",
            "admiral",
            "millenium",
            "chel",
            "pgrand",
            "tt",
            "sochi",
            "uap",
            "balans",
            "ta",
            "vrmsk"
    );

    private final Option<String> location;
    @BenderFlatten
    private final Resources resources;

    private EventLocation(Either<String, Resources> location) {
        this.location = location.leftO();
        this.resources = location.rightO().getOrElse(Resources::empty);
    }

    private EventLocation(Option<String> location, Resources resources) {
        this.location = location;
        this.resources = resources;
    }

    public String asTextForSms(Language lang) {
        return StringUtils.notEmptyO(resources.asTextForSms(lang)).plus(location).mkString("; ");
    }

    public String asTextForMailSubject(Language lang) {
        return StringUtils.notEmptyO(resources.asTextForMailSubject(lang)).plus(location).mkString(", ");
    }

    public void appendToXml(Element e) {
        Element eLocation = new Element("event-location");
        if (location.isPresent()) {
            EventXmlCreator.prepareAndAddIfNeeded(eLocation, "location", location.get());
        }
        if (resources.isNotEmpty()) {
            Element eResources = new Element("resources");
            resources.resources.forEach(r -> r.addXmlTo(eResources));
            eLocation.addContent(eResources);

            Element eDeleted = new Element("missed-resources");
            resources.missedResources.forEach(r -> r.addXmlTo(eDeleted));
            eLocation.addContent(eDeleted);

            CalendarXmlizer.appendElmColl(eLocation, "road-map-offices", "office", resources.roadMapOffices);
        }
        e.addContent(eLocation);
    }

    public static EventLocation compoundForPublic(
            String location, ListF<ResourceInfo> resources, ListF<ResourceInfo> missedResources, Language lang
    ) {
        return new EventLocation(
                StringUtils.notBlankO(location),
                new Resources(
                        consResources(resources, lang).map(Resource::withNameOnly),
                        consResources(missedResources, lang).map(Resource::withNameOnly),
                        Cf.list()
                ));
    }

    public static EventLocation location(String location) {
        return new EventLocation(Either.left(location));
    }

    public static EventLocation resources(
            ListF<ResourceInfo> resources, ListF<ResourceInfo> missedResources, Language lang)
    {
        return new EventLocation(Either.right(new Resources(
                consResources(resources, lang).map(Resource::withNameOnly),
                consResources(missedResources, lang).map(Resource::withNameOnly),
                ROAD_MAP_OFFICES.filter(resources.filterMap(r -> r.getOffice().getAbbr()).containsF()))));
    }

    public static EventLocation resourcesWithStaffMapLinkAndPhones(
            ListF<ResourceInfo> resources, ListF<ResourceInfo> missedResources, Language lang)
    {
        return new EventLocation(Either.right(new Resources(
                consResources(resources, lang),
                consResources(missedResources.filterNot(r -> r.getResource().getType() == ResourceType.CAMPUS), lang),
                ROAD_MAP_OFFICES.filter(resources.filterMap(r -> r.getOffice().getAbbr()).containsF()))));
    }

    private static final Function<Integer, NameI18n> WORKPLACES_I18N =
            NameI18n.plural("место", "места", "мест", "workplace", "workplaces");

    private static final NameI18n FLOOR_I18N = new NameI18n("этаж", "floor");

    private static ListF<Resource> consResources(ListF<ResourceInfo> resources, Language lang) {
        Tuple2<ListF<ResourceInfo>, ListF<ResourceInfo>> campusAndNot = resources
                .partition(r -> r.getResource().getType() == ResourceType.CAMPUS);

        ListF<Resource> campus = campusAndNot.get1()
                .groupBy(r -> Tuple3.tuple(r.getOfficeId(), r.getResource().getFloorNum(), r.getGroupNameI18n(lang)))
                .values().map(rs -> new Resource(
                        (rs.size() > 1 ? Option.of(WORKPLACES_I18N.apply(rs.size()).getName(lang)) : Option.<String>empty())
                                .plus(rs.first().getNameI18n(lang))
                                .plus(rs.first().getFloorNum().map(n -> n + " " + FLOOR_I18N.getName(lang)))
                                .plus(rs.first().getGroupNameI18n(lang))
                                .mkString(", "),
                        rs.first().getStaffMapLink(), Option.empty(), Option.empty()));

        ListF<Resource> others = campusAndNot.get2().filterMap(r -> {
            if (r.getNameWithAlterNameI18n().isPresent()) {
                return Option.of(new Resource(
                        r.getNameWithAlterNameI18n().get().getName(lang),
                        r.getStaffMapLink(), r.getPhone(), r.getVideo()));
            } else {
                return Option.empty();
            }
        });
        return campus.plus(others);
    }

    @BenderBindAllFields
    private static class Resource {
        private final String name;
        private final Option<String> href;
        private final Option<String> phone;
        private final Option<String> video;

        public Resource(String name, Option<String> href, Option<String> phone, Option<String> video) {
            this.name = name;
            this.href = href;
            this.phone = phone;
            this.video = video;
        }

        public String asTextForSms(Language lang) {
            Option<String> phone = this.phone.map(s -> UStringLiteral.TEL_LCI.getName(lang) + " " + s);
            Option<String> video = this.video.map(s -> UStringLiteral.VIDEO_LCI.getName(lang) + " " + s);

            return name + (phone.plus(video).isNotEmpty() ? ": " + phone.plus(video).mkString(", ") : ""); // CAL-6489
        }

        public void addXmlTo(Element e) {
            Element eResource = new Element("resource");
            for (Tuple2<String, String> valueName : Tuple2List.fromPairs(name, "name")
                    .plus(href.zip(Option.of("href")))
                    .plus(phone.zip(Option.of("phone")))
                    .plus(video.zip(Option.of("video"))))
            {
                CalendarXmlizer.appendElm(eResource, valueName.get2(), valueName.get1());
            }
            e.addContent(eResource);
        }

        public Resource withNameOnly() {
            return new Resource(name, Option.empty(), Option.empty(), Option.empty());
        }
    }

    @BenderBindAllFields
    private static class Resources {
        private final ListF<Resource> resources;
        private final ListF<Resource> missedResources;
        private final ListF<String> roadMapOffices;

        public Resources(
                ListF<Resource> resources, ListF<Resource> missedResources, ListF<String> roadMapOffices)
        {
            this.resources = resources;
            this.missedResources = missedResources;
            this.roadMapOffices = roadMapOffices;
        }

        public boolean isNotEmpty() {
            return !resources.isEmpty() || !missedResources.isEmpty();
        }

        public static Resources empty() {
            return new Resources(Cf.list(), Cf.list(), Cf.list());
        }

        public String asTextForSms(Language lang) {
            return resources.map(r -> r.asTextForSms(lang)).mkString("; ");
        }

        public String asTextForMailSubject(Language lang) {
            return resources.map(r -> r.withNameOnly().asTextForSms(lang)).mkString(", ");
        }
    }
}
