package ru.yandex.calendar.logic.sending.param;

import lombok.Getter;
import org.jdom.Element;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.contact.UnivContact;
import ru.yandex.calendar.logic.sending.EventXmlCreator;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.inside.passport.PassportSid;
import ru.yandex.misc.annotation.FillWithSomething;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.lang.StringUtils;

@BenderBindAllFields
public class EventMessageInfo {
    final long eventId;
    final long mainEventId;
    final Option<String> externalId;
    final Option<Instant> occurrenceId;

    @BenderFlatten
    final EventTimeParameters time;
    final String name;
    final String description;

    @Getter final EventLocation location;

    final Option<PassportSid> sid;
    final Option<UnivContact> organizer;

    final Option<Long> reorganizedEventId;

    @FillWithSomething
    public EventMessageInfo(
            long eventId, long mainEventId, String externalId, Option<Instant> occurrenceId,
            EventTimeParameters time, String name, String description,
            EventLocation location, PassportSid sid, Option<UnivContact> organizer, Option<Long> reorganizedEventId)
    {
        this.eventId = eventId;
        this.mainEventId = mainEventId;
        this.externalId = Option.of(externalId);
        this.occurrenceId = occurrenceId;
        this.time = time;
        this.name = name;
        this.description = description;
        this.location = location;
        this.sid = Option.when(!sid.sameAs(PassportSid.CALENDAR), sid);
        this.organizer = organizer;
        this.reorganizedEventId = reorganizedEventId;
    }

    public void appendXmlTo(Element r) {
        CalendarXmlizer.appendElm(r, "event-id", reorganizedEventId.getOrElse(eventId));
        CalendarXmlizer.appendElm(r, "event-name", this.name);
        CalendarXmlizer.appendElm(r, "event-start-ts", time.startTs);
        CalendarXmlizer.appendElm(r, "event-end-ts", time.endTs);

        EventXmlCreator.prepareAndAddIfNeeded(r, "event-description", this.description);
        CalendarXmlizer.appendElm(r, "show-is-all-day", time.showIsAllDay);

        if (time.repetition.isPresent()) {
            r.addContent(time.repetition.get().toXml("rep-hints"));
        }

        CalendarXmlizer.appendElm(r, "sid", this.sid.getOrElse(PassportSid.CALENDAR).getValue());
        CalendarXmlizer.appendElm(r, "is-meeting", this.organizer.isPresent());
        if (organizer.isPresent()) {
            String nick = StringUtils.notBlankO(organizer.get().getName())
                    .getOrElse(() -> Emails.getUnicoded(organizer.get().getEmail()));
            CalendarXmlizer.appendElm(r, "organizer-nick", nick);
        }
        location.appendToXml(r);
        time.emtz.appendToXml(r);
    }

    public LocalDateTime getEventStartTs() {
        return time.startTs;
    };

    public LocalDateTime getEventEndTs() {
        return time.endTs;
    }

    public String getEventName() {
        return name;
    }

    public String getEventDescription() {
        return description;
    }

    public long getEventId() {
        return eventId;
    }

    public boolean getIsRepeating() {
        return time.repetition.isPresent();
    }
}
