package ru.yandex.calendar.logic.sending.param;

import org.joda.time.DateTimeZone;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.contact.UnivContact;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.ChangedEventInfoForMails;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.ics.exp.EventInstanceParameters;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.sharing.participant.ParticipantBasics;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.participant.ResourceParticipantInfo;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.inside.passport.PassportSid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.misc.email.Email;

/**
 * @author gutman
 */
public class EventMessageInfoCreator {

    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;

    public EventMessageInfo create(
            EventWithRelations eventWr, RepetitionInstanceInfo repetition, EventInstanceParameters instance,
            Email recipientEmail, Option<PassportUid> recipientUid, Language language, DateTimeZone userTz)
    {
        return create(eventWr, repetition, instance, recipientEmail,
                recipientUid, Option.empty(), Option.empty(), Option.empty(), language, userTz);
    }

    public EventMessageInfo create(
            EventWithRelations eventWr, RepetitionInstanceInfo repetition, EventInstanceParameters instance,
            Email recipientEmail, Option<PassportUid> recipientUid,
            Option<ChangedEventInfoForMails> changes, Option<ParticipantBasics> organizerOverride,
            Option<Long> reorganizedEventId, Language language, DateTimeZone userTz)
    {
        Event event = eventWr.getEvent();
        Participants participants = eventWr.getParticipants();

        EventTimeParameters time = EventTimeParameters.create(eventWr.getTime(), instance, repetition, userTz);

        ListF<ResourceInfo> resources = participants.getParticipantsSafe()
                .filterByType(ResourceParticipantInfo.class).map(ResourceParticipantInfo.resourceInfoF());

        ListF<Long> survivedOfficeIds = resources.map(ResourceInfo::getOfficeId);
        ListF<ResourceInfo> missedRss = changes.flatMap(c -> c.getChanges().getRemovedResources()
                .filterNot(r -> survivedOfficeIds.containsTs(r.getOfficeId())));

        if (recipientUid.isPresent() && resources.isNotEmpty()) {
            resources = resourceRoutines.sortResourcesFromUserOfficeAndCityFirst(recipientUid.get(), resources);
        }
        if (recipientUid.isPresent() && missedRss.isNotEmpty()) {
            missedRss = resourceRoutines.sortResourcesFromUserOfficeAndCityFirst(recipientUid.get(), missedRss);
        }

        final EventLocation location;

        if (!passportAuthDomainsHolder.containsYandexTeamRu()) {
            location = EventLocation.compoundForPublic(event.getLocation(), resources, missedRss, language);
        } else if (resources.isEmpty() && (missedRss.isEmpty() || !event.getLocation().isEmpty())) {
            location = EventLocation.location(event.getLocation());
        } else if (PassportAuthDomain.byEmail(recipientEmail) == PassportAuthDomain.YANDEX_TEAM_RU) {
            location = EventLocation.resourcesWithStaffMapLinkAndPhones(resources, missedRss, language);
        } else {
            location = EventLocation.resources(resources, missedRss, language);
        }

        Option<UnivContact> organizer = Option.empty();
        if (participants.isMeeting()) {
            ParticipantBasics p = Option.<ParticipantBasics>empty()
                    .orElse(organizerOverride).getOrElse(participants::getOrganizer);

            NameI18n nameI18n = eventInvitationManager.getParticipantNameI18ns(Cf.list(p)).single().getNameI18n();

            organizer = Option.of(new UnivContact(p.getEmail(), nameI18n.getName(language), p.getUid()));
        }

        PassportSid sid = event.getCreationSource() == ActionSource.MAIL ? PassportSid.MAIL : event.getSid();

        return new EventMessageInfo(
                event.getId(), event.getMainEventId(), eventWr.getExternalId(), instance.getOccurrenceId(),
                time, event.getName(), event.getDescription(), location, sid, organizer, reorganizedEventId);
    }

}
