package ru.yandex.calendar.logic.sending.param;

import org.jdom.Element;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.ics.iv5j.ical.PropertyNames;
import ru.yandex.calendar.logic.log.EventIdLogDataJson;
import ru.yandex.calendar.logic.sending.bazinga.MessageExtraDao;
import ru.yandex.calendar.logic.sharing.MailType;
import ru.yandex.commune.mail.MailAddress;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;

public abstract class EventMessageParameters extends MessageParameters {

    // TODO: extract real common part (sender, recipient& etc) and create common
    // with LayerInvitationMessageParameters parent class.
    protected final CommonEventMessageParameters meta;
    protected final EventMessageInfo event;
    protected Option<IcsCalendar> ics;

    public EventMessageParameters(CommonEventMessageParameters meta, EventMessageInfo event) {
        this(meta, event, Option.empty());
    }

    public EventMessageParameters(CommonEventMessageParameters meta,
                                  EventMessageInfo event,
                                  Option<IcsCalendar> ics) {
        this.meta = meta;
        this.event = event;
        this.ics = ics;
    }

    @Override
    public String toString() {
        return getClass().getSimpleName() + "{"
                + "id=" + event.eventId
                + ", to=" + meta.getRecipientAddress().getEmail()
                + "}";
    }

    public Option<PassportUid> getRecipientUid() {
        return meta.getRecipientUid();
    }

    public MailAddress getRecipientAddress() {
        return meta.getRecipientAddress();
    }

    public static Function<EventMessageParameters, Email> getRecipientEmailF() {
        return MessageParameters::getRecipientEmail;
    }

    public static Function<EventMessageParameters, Option<PassportUid>> getRecipientUidF() {
        return eventMessageParameters -> eventMessageParameters.meta.getRecipientUid();
    }

    public static Function<EventMessageParameters, MailType> mailTypeF() {
        return MessageParameters::mailType;
    }

    public static Function<EventMessageParameters, Long> getEventIdF() {
        return p -> p.event.eventId;
    }

    public Sender getSender() {
        return meta.getSender();
    }

    public long getEventId() {
        return event.eventId;
    }

    public long getMainEventId() {
        return event.mainEventId;
    }

    public Option<Instant> getOccurrenceId() {
        return event.occurrenceId;
    }

    public EventMessageInfo getEventMessageInfo() {
        return event;
    }

    @Override
    public Element toOldStyleXml() {
        Element r = new Element("message");

        meta.appendXmlTo(r);
        event.appendXmlTo(r);
        return r;
    }

    public EventIdLogDataJson getEventIdLogDataJson() {
        return new EventIdLogDataJson(event.eventId, event.mainEventId, event.externalId.getOrElse(""), event.occurrenceId.toOptional());
    }

    public EventIdLogDataJson getEventIdLogData() {
        return new EventIdLogDataJson(event.eventId, event.mainEventId, event.externalId.getOrElse(""), event.occurrenceId.toOptional());
    }

    @Override
    public Recipient getRecipient() {
        return meta.getRecipient();
    }

    @Override
    public MessageOverrides getMessageOverrides() {
        return meta.getOverrides();
    }

    @Override
    public void loadExtra(MessageExtraDao messageExtraDao) {
        ics.ifPresent(presentIcs -> {
            var updatedComponents = presentIcs.getComponents().map(component -> {
                return component.getPropertyValue(PropertyNames.X_YANDEX_MESSAGE_EXTRA_ID)
                        .map(Long::parseLong)
                        .flatMapO(messageExtraDao::load)
                        .map(extra -> extra.properties.getProperties())
                        .map(component::addProperties)
                        .map(updatedComponent -> updatedComponent.removeProperties(PropertyNames.X_YANDEX_MESSAGE_EXTRA_ID))
                        .getOrElse(component);
            });
            ics = Option.of(new IcsCalendar(updatedComponents, presentIcs.getProperties()));
        });
    }

    @Override
    public MessageParameters withOverrides(MessageOverrides overrides) {
        return withMeta(meta.withOverrides(overrides));
    }

    public abstract MessageParameters withMeta(CommonEventMessageParameters meta);

    public Option<IcsCalendar> getIcs() {
        return ics;
    }
}
