package ru.yandex.calendar.logic.sending.param;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.event.EventInterval;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventTime;
import ru.yandex.calendar.logic.event.repetition.RepetitionHints;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.ics.exp.EventInstanceParameters;
import ru.yandex.misc.annotation.FillWithSomething;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;

/**
 * @author dbrylev
 */
@BenderBindAllFields
public class EventTimeParameters {
    final LocalDateTime startTs;
    final LocalDateTime endTs;
    final boolean isAllDay;
    final boolean showIsAllDay;

    final Option<RepetitionHints> repetition;
    final EventMessageTimezone emtz;

    @FillWithSomething
    public EventTimeParameters(
            LocalDateTime startTs, LocalDateTime endTs, boolean isAllDay, boolean showIsAllDay,
            Option<RepetitionHints> repetition, EventMessageTimezone emtz)
    {
        this.startTs = startTs;
        this.endTs = endTs;
        this.isAllDay = isAllDay;
        this.showIsAllDay = showIsAllDay;
        this.repetition = repetition;
        this.emtz = emtz;
    }

    public static EventTimeParameters create(
            EventTime time, EventInstanceParameters instance,
            RepetitionInstanceInfo repetition, DateTimeZone userTz)
    {
        EventInterval interval;
        Option<RepetitionHints> repetitionHints;

        if (!instance.getOccurrenceId().isPresent() && repetition.getRepetition().isPresent()) {
            interval = time.getEventInterval(RepetitionUtils.getInstanceIntervalEndingAfter(
                    repetition.withoutRdatesAndRecurrences(), Instant.now()).getOrElse(instance.getInterval()));

            repetitionHints = repetition.createHints();

        } else {
            interval = time.getEventInterval(instance.getInterval());
            repetitionHints = Option.empty();
        }

        LocalDateTime startLocal = interval.getStart().toLocalDateTime(userTz);
        LocalDateTime endLocal = interval.getEnd().toLocalDateTime(userTz);

        EventMessageTimezone emtz = EventMessageTimezone.create(userTz, interval.getInstanceStart());

        return new EventTimeParameters(
                startLocal, endLocal, time.isAllDay(),
                EventRoutines.getShowIsAllDay(time, userTz), repetitionHints, emtz);
    }
}
